/**
 * This work was created by participants in the DataONE project, and is
 * jointly copyrighted by participating institutions in DataONE. For 
 * more information on DataONE, see our web site at http://dataone.org.
 *
 *   Copyright ${year}
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and 
 * limitations under the License.
 * 
 * $Id$
 */

package gov.ornl.mercury3.web.util;

import gov.ornl.mercury3.commands.Configuration;

import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.dataone.client.CNode;
import org.dataone.service.exceptions.NotImplemented;
import org.dataone.service.exceptions.ServiceFailure;
import org.dataone.service.types.v1.Node;
import org.dataone.service.types.v1.NodeType;
import org.springframework.context.ApplicationContext;
import org.springframework.context.support.ClassPathXmlApplicationContext;

/**
 * Singleton pattern cache for member nodes in d1 cluster.
 * 
 * Used by MNSourcesTag to generate html to create an option tags for input
 * select control.
 * 
 * @author sroseboo
 * 
 */
public class MemberNodeSourcesMapCache {

    private static final String NODE_LIST_URL_KEY = "node_list_url";
    private static String NODE_LIST_URL = "https://localhost/cn/";

    private static final String CACHE_EXPIRATION_KEY = "cache_expiration_period";
    private static long CACHE_EXPIRATION_PERIOD = 60000 * 5; // 5 minutes

    private static final String NODE_STALE_SYNC_DATE_KEY = "node_sync_stale_date";
    private static String STALE_SYNC_DATE_STRING = "2012/07/01";
    private static Date STALE_SYNC_DATE = null;

    private Map<String, String> mnSourceMap = new HashMap<String, String>();
    private long mnSourceCacheTimestamp = 0;

    public static final MemberNodeSourcesMapCache INSTANCE = new MemberNodeSourcesMapCache();

    private MemberNodeSourcesMapCache() {
        configureProperties();
    }

    public Map<String, String> getMNSourceMap() {
        if (timeToRefresh()) {
            mnSourceMap = refreshMNSourceMap();
        }
        return mnSourceMap;
    }

    private synchronized Map<String, String> refreshMNSourceMap() {
        Map<String, String> freshMap = new HashMap<String, String>();
        try {

            CNode cn = new CNode(NODE_LIST_URL);
            List<Node> values = cn.listNodes().getNodeList();
            for (Node node : values) {
                if (isMemberNode(node) && isCurrentSync(node)) {
                    freshMap.put(node.getIdentifier().getValue(), node.getName());
                }
            }
        } catch (ServiceFailure sf) {
            sf.printStackTrace();
        } catch (NotImplemented ni) {
            ni.printStackTrace();
        }
        mnSourceCacheTimestamp = System.currentTimeMillis();
        return freshMap;
    }

    private boolean timeToRefresh() {
        return mnSourceCacheTimestamp + CACHE_EXPIRATION_PERIOD < System.currentTimeMillis();
    }

    private boolean isMemberNode(Node node) {
        return node.getType().compareTo(NodeType.MN) == 0;
    }

    private boolean isCurrentSync(Node node) {
        if (STALE_SYNC_DATE == null) {
            return true;
        }
        if (node.getSynchronization() == null
                || node.getSynchronization().getLastHarvested() == null) {
            return false;
        }
        return STALE_SYNC_DATE.before(node.getSynchronization().getLastHarvested());
    }

    private void configureProperties() {
        ApplicationContext factory = new ClassPathXmlApplicationContext("Mercury3Properties.xml");
        Configuration cv = (Configuration) factory.getBean("propertiesBean");
        HashMap<String, Object> hmProps = cv.getProperties();
        String nodeUrlString = (String) hmProps.get(NODE_LIST_URL_KEY);
        if (null != nodeUrlString && !"".equals(nodeUrlString)) {
            NODE_LIST_URL = nodeUrlString.trim();
        }
        String cacheExpirationString = (String) hmProps.get(CACHE_EXPIRATION_KEY);
        if (null != cacheExpirationString && !"".equals(cacheExpirationString)) {
            CACHE_EXPIRATION_PERIOD = Long.valueOf(cacheExpirationString.trim()).longValue();
        }
        String syncStaleDate = (String) hmProps.get(NODE_STALE_SYNC_DATE_KEY);
        if (syncStaleDate != null && !"".equals(syncStaleDate)) {
            STALE_SYNC_DATE_STRING = syncStaleDate;
        }
        try {
            DateFormat dateFormat = new SimpleDateFormat("yyyy/MM/dd");
            STALE_SYNC_DATE = dateFormat.parse(STALE_SYNC_DATE_STRING);
        } catch (ParseException pe) {
            System.out.println("Failed to parse stale sync date: " + STALE_SYNC_DATE_STRING);
            pe.printStackTrace();
        }
    }
}
