/**
 * This work was created by participants in the DataONE project, and is
 * jointly copyrighted by participating institutions in DataONE. For 
 * more information on DataONE, see our web site at http://dataone.org.
 *
 *   Copyright ${year}
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and 
 * limitations under the License.
 * 
 * $Id$
 */

package gov.ornl.mercury3.services;

import gov.ornl.mercury3.commands.CitationDateDisplayUtil;
import gov.ornl.mercury3.commands.CoinsBean;
import gov.ornl.mercury3.commands.Configuration;
import gov.ornl.mercury3.commands.Field;
import gov.ornl.mercury3.commands.FilterCatField;
import gov.ornl.mercury3.commands.Response;
import gov.ornl.mercury3.email_utils.EmailGenerator;
import gov.ornl.mercury3.web.util.D1DocTransferObject;
import gov.ornl.mercury3.web.util.MemberNodeSourcesMapCache;

import java.io.BufferedReader;
import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.PrintWriter;
import java.io.Reader;
import java.io.StringWriter;
import java.io.Writer;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLEncoder;
import java.text.ParsePosition;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;

import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.Source;
import javax.xml.transform.SourceLocator;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerConfigurationException;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.stream.StreamResult;
import javax.xml.transform.stream.StreamSource;

import org.apache.commons.httpclient.NameValuePair;
import org.apache.commons.httpclient.util.EncodingUtil;
import org.apache.commons.io.IOUtils;
import org.apache.solr.client.solrj.util.ClientUtils;
import org.dataone.client.D1RestClient;
import org.dataone.service.types.v1.Session;
import org.jdom.Document;
import org.jdom.Element;
import org.jdom.JDOMException;
import org.jdom.input.SAXBuilder;
import org.jdom.xpath.XPath;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import org.springframework.context.ApplicationContext;
import org.springframework.context.support.ClassPathXmlApplicationContext;
import org.xml.sax.SAXException;

/**
 * Primary class for building final SOLR queries and handling the SOLR response.
 * 
 */
public class SolrSearcher {

    private boolean spellCheck = false;

    public SimpleMapBean mlt_bean = new SimpleMapBean();
    public DataSourceMapNamesBean dsmnb = new DataSourceMapNamesBean();
    public FilterslistMapBean flmb = new FilterslistMapBean();

    private String limitlessField = "";
    private String limitlessFieldLimit = "";
    private boolean isRSS = false;
    private boolean isMLT = false;
    private String spell = "";
    private final String COMMON = "common";
    private String feedValue = COMMON;
    private String toEMAIL = "";
    private String fromEMAIL = "";
    private String source = "";

    private String shards = "";

    // Filter query string passed in from configuration
    private String publicFqPart = "";

    private ArrayList<HashMap<String, String>> mlt_list = new ArrayList<HashMap<String, String>>();

    String url = null;
    ApplicationContext factory3 = null;
    public Map<String, String> datasourceMapNames = new HashMap();
    public HashMap<String, String> filterslist = new HashMap();

    public HashMap<String, String> mlt_terms = new HashMap();

    public SimpleDateFormat Solr_ISO8601FORMAT = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ssZ");

    public SimpleDateFormat Brief_Format = new SimpleDateFormat("MM/dd/yyyy");

    public SimpleDateFormat Solr_RFC822DATEFORMAT = new SimpleDateFormat(
            "EEE', 'dd' 'MMM' 'yyyy' 'HH:mm:ss' 'Z", Locale.US);

    public SimpleDateFormat PRT_Format = new SimpleDateFormat("yyyy-MM-dd");

    public String convertToBrief(String date) {
        try {
            Solr_ISO8601FORMAT.parse(date, new ParsePosition(0));
        } catch (Exception pex) {
            pex.printStackTrace();
        }
        return Brief_Format.format(Solr_ISO8601FORMAT.getCalendar().getTime());
    }

    public String convertToPRT(String date) {
        try {
            Brief_Format.parse(date, new ParsePosition(0));
        } catch (Exception pex) {
            pex.printStackTrace();
        }
        return PRT_Format.format(Brief_Format.getCalendar().getTime());
    }

    public SolrSearcher(final String _url) throws MalformedURLException {

        new URL(_url);

        factory3 = new ClassPathXmlApplicationContext(
                "gov/ornl/mercury3/services/Mercury_term_maps.xml");

        dsmnb = (DataSourceMapNamesBean) factory3.getBean("datasource_map_names");
        datasourceMapNames = dsmnb.getDatasourceMapNames();

        flmb = (FilterslistMapBean) factory3.getBean("Filterslist_map");
        filterslist = flmb.getFilterslist();
        url = _url;

        ApplicationContext factory = new ClassPathXmlApplicationContext("Mercury3Properties.xml");//
        Configuration cv = (Configuration) factory.getBean("propertiesBean");
        HashMap hmProps = cv.getProperties();

        try {
            toEMAIL = (String) hmProps.get("toEMAIL");
            fromEMAIL = (String) hmProps.get("fromEMAIL");
            shards = (String) hmProps.get("shards");

        } catch (Exception exc) {
            exc.printStackTrace();
            toEMAIL = "mercury-support@ornl.gov";
            fromEMAIL = "palanisamyg@ornl.gov";
        }
        try {
            if (null != hmProps.get("spellCheck")) {
                spellCheck = Boolean.parseBoolean((String) hmProps.get("spellCheck"));
            }
        } catch (Exception exc) {
            exc.printStackTrace();
        }

        try {
            shards = (String) hmProps.get("shards");
        } catch (Exception exc) {
            exc.printStackTrace();
        }

        if ((null == toEMAIL) || (toEMAIL.length() == 0)) {
            toEMAIL = "mercury-support@ornl.gov";
        }

        if ((null == fromEMAIL) || (fromEMAIL.length() == 0)) {
            fromEMAIL = "palanisamyg@ornl.gov";
        }

        boolean public_only = Boolean.parseBoolean((String) (hmProps.get("show_public_only")));
        if (public_only) {
            publicFqPart = (String) hmProps.get("public_fq_part");
        }
        boolean use_mn_source_cache = Boolean.parseBoolean((String) (hmProps
                .get("use_mn_source_cache")));
        if (use_mn_source_cache) {
            datasourceMapNames = MemberNodeSourcesMapCache.INSTANCE.getMNSourceMap();
        }

    }

    public Response search(final String _query, final int _start, final int _rows,
            final List<String> _fields, final String _type, final String _schema,
            final String _dismaxSort, final String _facet, final String _facetLimit,
            final List<String> _facetField, String imitlessFacet, String limtlessFieldValue,
            boolean isRSS, String feedValue, boolean isMLT, String spell,
            List<String> queryFilters, Session d1Session) {

        limitlessField = imitlessFacet;
        limitlessFieldLimit = limtlessFieldValue;
        return search(_query, _start, _rows, _fields, _type, _schema, _dismaxSort, _facet,
                _facetLimit, _facetField, isRSS, feedValue, isMLT, spell, queryFilters, d1Session);
    }

    public Response search(final String _query, final int _start, final int _rows,
            final List<String> _fields, final String _type, final String _schema,
            final String _dismaxSort, final String _facet, final String _facetLimit,
            final List<String> _facetField, List<String> queryFilters, Session d1Session) {

        isRSS = false;
        isMLT = false;
        spell = "";
        return search(_query, _start, _rows, _fields, _type, _schema, _dismaxSort, _facet,
                _facetLimit, _facetField, isRSS, feedValue, isMLT, spell, queryFilters, d1Session);

    }

    public List<D1DocTransferObject> findRelatedDocByResourceMap(String pid, Session d1Session) {

        List<D1DocTransferObject> results = new ArrayList<D1DocTransferObject>();
        final List<NameValuePair> params = new ArrayList<NameValuePair>();

        params.add(new NameValuePair("q", "isDocumentedBy:" + ClientUtils.escapeQueryChars(pid)));
        params.add(new NameValuePair("fl",
                "id,dataUrl,isPublic,size,formatId,formatType,resourceMap"));
        params.add(new NameValuePair("sort", "formatType desc,formatId asc,size desc"));
        params.add(new NameValuePair("wt", "json"));

        params.add(new NameValuePair("rows", "1000"));

        String paramString = EncodingUtil.formUrlEncode(params.toArray(new NameValuePair[0]),
                "UTF-8");
        try {
            D1RestClient restClient = new D1RestClient(d1Session);
            InputStream response = restClient.doGetRequest(url + "/?" + paramString);
            final String responseBody = IOUtils.toString(response, "UTF-8");

            JSONObject resultObject = new JSONObject(responseBody);
            if (resultObject.has("response")) {
                JSONArray docs = resultObject.getJSONObject("response").getJSONArray("docs");
                for (int i = 0; i < docs.length(); i++) {
                    JSONObject result = docs.getJSONObject(i);
                    D1DocTransferObject d1Doc = new D1DocTransferObject(result.getString("id"),
                            result.getString("dataUrl"), result.getString("formatId"),
                            result.getString("size"), result.getString("isPublic"));
                    if (result.has("resourceMap")) {
                        d1Doc.setResourceMap(result.getJSONArray("resourceMap"));
                    }
                    if ("METADATA".equals(result.getString("formatType"))) {
                        d1Doc.setMetadata(true);
                    }
                    results.add(d1Doc);
                }
            }
        } catch (Exception e) {
            System.out.println("Error in Solr Searcher findRelatedD1Docs: ");
            e.printStackTrace();
        }
        return results;
    }

    public D1DocTransferObject getD1DocInfo(String pid, Session d1Session) {

        D1DocTransferObject d1DocDto = null;
        final List<NameValuePair> params = new ArrayList<NameValuePair>();

        params.add(new NameValuePair("q", "id:" + ClientUtils.escapeQueryChars(pid)));
        params.add(new NameValuePair("fl",
                "id,dataUrl,isPublic,size,formatId,formatType,resourceMap"));
        params.add(new NameValuePair("wt", "json"));

        String paramString = EncodingUtil.formUrlEncode(params.toArray(new NameValuePair[0]),
                "UTF-8");
        try {
            D1RestClient restClient = new D1RestClient(d1Session);
            InputStream response = restClient.doGetRequest(url + "/?" + paramString);
            final String responseBody = IOUtils.toString(response, "UTF-8");

            JSONObject resultObject = new JSONObject(responseBody);
            if (resultObject.has("response")) {
                JSONArray docs = resultObject.getJSONObject("response").getJSONArray("docs");
                if (docs.length() == 1) {
                    JSONObject result = docs.getJSONObject(0);
                    d1DocDto = new D1DocTransferObject(result.getString("id"),
                            result.getString("dataUrl"), result.getString("formatId"),
                            result.getString("size"), result.getString("isPublic"));
                    if ("METADATA".equals(result.getString("formatType"))) {
                        d1DocDto.setMetadata(true);
                    }
                    if (result.has("resourceMap")) {
                        d1DocDto.setResourceMap(result.getJSONArray("resourceMap"));
                    }
                }
            }
        } catch (Exception e) {
            System.out.println("Error in SolrSearcher getD1DocInfo method: ");
            e.printStackTrace();
        }
        return d1DocDto;
    }

    public CoinsBean getCoinsData(String pid, Session d1Session) {

        CoinsBean coins = new CoinsBean();
        final List<NameValuePair> params = new ArrayList<NameValuePair>();

        params.add(new NameValuePair("q", "id:" + ClientUtils.escapeQueryChars(pid)));
        params.add(new NameValuePair(
                "fl",
                "id, dataUrl, title, abstract, datasource, author, pubDate, beginDate, dateUploaded, keywords, investigator"));
        params.add(new NameValuePair("wt", "json"));

        String paramString = EncodingUtil.formUrlEncode(params.toArray(new NameValuePair[0]),
                "UTF-8");
        try {
            D1RestClient restClient = new D1RestClient(d1Session);
            InputStream response = restClient.doGetRequest(url + "/?" + paramString);
            final String responseBody = IOUtils.toString(response, "UTF-8");

            JSONObject resultObject = new JSONObject(responseBody);
            if (resultObject.has("response")) {
                JSONArray docs = resultObject.getJSONObject("response").getJSONArray("docs");
                if (docs.length() == 1) {
                    JSONObject result = docs.getJSONObject(0);
                    coins.setPid(getJsonString(result, "id"));
                    coins.setDataUrl(getJsonString(result, "dataUrl"));
                    coins.setTitle(getJsonString(result, "title"));
                    coins.setAbstractText(getJsonString(result, "abstract"));
                    coins.setDatasource(getJsonString(result, "datasource"));
                    coins.setAuthor(getJsonString(result, "author"));

                    String citationDate = CitationDateDisplayUtil.getCitationDate(
                            getJsonString(result, "pubDate"), getJsonString(result, "beginDate"),
                            getJsonString(result, "dateUploaded"));
                    coins.setCitationDate(citationDate);

                    if (result.has("keywords")) {
                        coins.setKeywordsFromJsonArray(result.getJSONArray("keywords"));
                    }

                    if (result.has("investigator")) {
                        coins.setAdditionalInvestigatorsFromJsonArray(result
                                .getJSONArray("investigator"));
                    }
                }
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return coins;
    }

    public String solrPass(Map parameters, Session d1Session) {
        String responseBody = "";

        final List<NameValuePair> params = new ArrayList<NameValuePair>();
        for (Object key : parameters.keySet()) {
            String keyString = (String) key;
            String[] values = (String[]) parameters.get(keyString);
            if (values != null) {
                for (int i = 0; i < values.length; i++) {
                    String value = values[i];
                    params.add(new NameValuePair(keyString, value));
                }
            }
        }
        String paramString = EncodingUtil.formUrlEncode(params.toArray(new NameValuePair[0]),
                "UTF-8");
        try {
            D1RestClient restClient = new D1RestClient(d1Session);
            InputStream response = restClient.doGetRequest(url + "/?" + paramString);
            responseBody = IOUtils.toString(response, "UTF-8");
        } catch (Exception e) {
            e.printStackTrace();
        }
        return responseBody;
    }

    private String getJsonString(JSONObject result, String fieldName) {
        String value = "";
        if (result.has(fieldName)) {
            try {
                return result.getString(fieldName);
            } catch (JSONException jse) {
                jse.printStackTrace();
            }
        }
        return value;
    }

    public Response search(final String _query, final int _start, final int _rows,
            final List<String> _fields, final String _type, final String _schema,
            final String _dismaxSort, final String _facet, final String _facetLimit,
            final List<String> _facetField, boolean isRSS, String feedValue, boolean isMLT,
            String spell, List<String> queryFilters, Session d1Session) {

        final List<NameValuePair> params = new ArrayList<NameValuePair>();

        String q3 = _query;

        String q2 = _query;

        if (q2.trim().replaceAll(" ", "").equalsIgnoreCase("((*))")) {
            q2 = "((*:*))";
        }
        if (q2.trim().replaceAll(" ", "").contains("(*)")) {
            q2 = _query.trim().replaceAll("( \\* )", "(\\*:\\*)");

        }
        if (feedValue.equalsIgnoreCase("gfis")) {
            q3 = q2.concat(" AND pubDate:[NOW/YEAR-10YEAR TO * ]");
        }

        params.add(new NameValuePair("q", q3));

        params.add(new NameValuePair("spellcheck.q", spell));
        params.add(new NameValuePair("spellcheck", "true"));
        params.add(new NameValuePair("spellcheck.collate", "true"));

        if (isMLT) {
            mlt_bean = (SimpleMapBean) factory3.getBean("MLT_Map");
            mlt_terms = mlt_bean.getTerm1Map();
            for (String var : mlt_terms.keySet()) {
                params.add(new NameValuePair(var, mlt_terms.get(var)));
            }
        }

        if (_dismaxSort != null) {
            params.add(new NameValuePair("sort", _dismaxSort));
        }
        if (_start > 0) {
            params.add(new NameValuePair("start", _start + ""));
        }
        params.add(new NameValuePair("rows", _rows + ""));
        if (_type != null) {
            params.add(new NameValuePair("qt", _type));
        }
        if (_schema != null) {
            params.add(new NameValuePair("df", _schema));
        }
        if ((shards != null) && (shards.length() > 0)) {
            params.add(new NameValuePair("shards", shards));
        }
        if (_fields != null) {
            final StringBuilder sb = new StringBuilder();
            for (String string : _fields) {
                sb.append(string);
                sb.append(",");
            }
            params.add(new NameValuePair("fl", sb.toString()));
        }

        if (_facet != null)
            params.add(new NameValuePair("facet", "true"));
        if (_facetLimit != null)
            params.add(new NameValuePair("facet.limit", _facetLimit));
        int order = Integer.parseInt(_facetLimit);
        if (order == -1) {
            params.add(new NameValuePair("facet.sort", "false"));
        }
        if (_facetField != null) {
            final StringBuilder sb2 = new StringBuilder();
            for (String string : _facetField) {
                params.add(new NameValuePair("facet.field", string));
            }
        }

        params.add(new NameValuePair("facet.zeros", "false"));

        if (limitlessField.length() > 0 && limitlessFieldLimit.length() > 0) {
            params.add(new NameValuePair("f." + limitlessField + ".facet.limit",
                    limitlessFieldLimit));
        }

        if (publicFqPart != null && publicFqPart.isEmpty() == false) {
            params.add(new NameValuePair("fq", publicFqPart));
        }

        if (queryFilters != null && queryFilters.isEmpty() == false) {
            for (String queryFilter : queryFilters) {
                params.add(new NameValuePair("fq", queryFilter));
            }
        }

        String encoded = EncodingUtil.formUrlEncode(params.toArray(new NameValuePair[0]), "UTF-8");
        String requestUrl = url + "/?" + encoded;
        Response res = new Response();

        try {
            D1RestClient restClient = new D1RestClient(d1Session);
            InputStream response = restClient.doGetRequest(requestUrl);
            final String responseBody = IOUtils.toString(response, "UTF-8");
            res = createResponse(responseBody, _fields);

            if (isRSS) {
                ApplicationContext factory = new ClassPathXmlApplicationContext(
                        "gov/ornl/mercury3/services/applicationContext.xml");
                File xslFilename = null;
                try {
                    if (feedValue.equalsIgnoreCase(COMMON) || (feedValue.length() < 1)) {
                        xslFilename = factory.getResource(
                                "gov/ornl/mercury3/services/example_rss.xsl").getFile();
                    } else {
                        xslFilename = factory.getResource(
                                "gov/ornl/mercury3/services/" + feedValue + "_rss.xsl").getFile();
                    }
                    String inputString = getCleanString(responseBody);
                    final InputStream is = new ByteArrayInputStream(inputString.getBytes(("UTF-8")));

                    String rss1 = getTransform(xslFilename, is);
                    res.setRss1(rss1);
                } catch (Exception e) {
                    e.printStackTrace();
                }
            }

            if (isMLT) {
                ApplicationContext factory = new ClassPathXmlApplicationContext(
                        "gov/ornl/mercury3/services/applicationContext.xml");
                File xslFilename = null;
                try {
                    xslFilename = factory.getResource("gov/ornl/mercury3/services/mlt.xsl")
                            .getFile();
                    String inputString = getCleanString(responseBody);
                    final InputStream is = new ByteArrayInputStream(inputString.getBytes(("UTF-8")));
                    String rss1 = getTransform(xslFilename, is);
                    res.setRss1(rss1);
                } catch (Exception e) {
                    e.printStackTrace();
                }
            }
        } catch (Exception ex) {
            System.out.println("**** Error in SolrSearch search method: ");
            ex.printStackTrace();
        }
        return res;

    }

    public String getTransform(File xslFilename, InputStream is) {

        File xsltFile = xslFilename;
        StreamResult strResult = new StreamResult(new StringWriter());
        String rss = null;
        try {

            Source xmlSource3 = new StreamSource(is);
            Source xsltSource = new StreamSource(xsltFile);
            TransformerFactory transFact = TransformerFactory.newInstance();
            Transformer trans = transFact.newTransformer(xsltSource);
            trans.transform(xmlSource3, strResult);

            rss = strResult.getWriter().toString();

        } catch (TransformerConfigurationException e) {
            e.printStackTrace();
            System.out.println("An error occurred in the XSL file" + e);
        } catch (TransformerException e) {
            e.printStackTrace();
            SourceLocator locator = e.getLocator();
            int col = locator.getColumnNumber();
            int line = locator.getLineNumber();
            System.out.println(" An error occurred while applying the XSL file");
            System.out.println("line : " + line + "  col : " + col);
        }

        return rss;
    }

    /**
     * @param _xml
     * @param _fields
     * @return
     * @throws SAXException
     * @throws IOException
     * @throws ParserConfigurationException
     * @throws JDOMException
     */
    protected Response createResponse(final String _xml, final List<String> _fields)
            throws SAXException, IOException, ParserConfigurationException, JDOMException {

        final Response res = new Response();
        final byte xmlBytes[] = _xml.getBytes("UTF-8");
        final InputStream is = new ByteArrayInputStream(xmlBytes);

        StringBuffer buffer = new StringBuffer();
        InputStreamReader isr2 = new InputStreamReader(is, "UTF-8");
        Reader in2 = new BufferedReader(isr2);
        int ch;

        while ((ch = in2.read()) > -1) {
            if ((ch == 0x9) || (ch == 0xA) || (ch == 0xD) || ((ch >= 0x20) && (ch <= 0xD7FF))
                    || ((ch >= 0xE000) && (ch <= 0xFFFD)) || ((ch >= 0x10000) && (ch <= 0x10FFFF))) {
                buffer.append((char) ch);
            } else {
                System.out.println("Filtering characters = > " + ch);
                System.out.println("");
            }

        }
        in2.close();

        String myString = buffer.toString();
        final byte xmlBytes2[] = myString.getBytes("UTF-8");
        final InputStream is2 = new ByteArrayInputStream(xmlBytes2);
        final Document doc = new SAXBuilder().build(is2);

        // //////////////////
        // Experimental block for extracting a spell checker suggestion
        // bool derived from config spellCheck
        if (spellCheck) {
            String xexprBean = "//response/ lst[@name='spellcheck']/ lst[@name='suggestions']/ str[@name='collation']/text()";
            XPath beans = XPath.newInstance(xexprBean);
            String suggestion = beans.valueOf(doc);
            res.setSuggestion(suggestion);
            System.out.println("SpellChecker suggestion => " + suggestion);
        }

        // ///////////////////
        final Element root = doc.getRootElement();
        List responseHeaderList = root.getChildren("lst");

        Iterator rhLIT = responseHeaderList.iterator();

        while (rhLIT.hasNext()) {

            Element rDocument = (Element) rhLIT.next();

            if (rDocument.getAttributeValue("name").equals("responseHeader")) {
                Iterator rDIT = (rDocument.getChildren("int")).iterator();

                while (rDIT.hasNext()) {
                    Element intElm = (Element) rDIT.next();

                    if (intElm.getAttributeValue("name").equals("status")) {
                        res.setStatus(new Integer(intElm.getValue()));
                    }
                    if (intElm.getAttributeValue("name").equals("QTime")) {
                        res.setQtime(new Integer(intElm.getValue()));
                    }
                }
            }
            /*
             * Parses the solr similarity response and returns it as a list of
             * maps for the brief page.
             */
            else if (rDocument.getAttributeValue("name").equals("moreLikeThis")) {
                Iterator rDIT = (rDocument.getChildren("result")).iterator();
                while (rDIT.hasNext()) {
                    Element intElm = (Element) rDIT.next();
                    Iterator rDIT2 = (intElm.getChildren("doc")).iterator();
                    while (rDIT2.hasNext()) {
                        HashMap<String, String> mlt_hm = new HashMap<String, String>();
                        Element intElm2 = (Element) rDIT2.next();
                        Iterator rDIT3 = (intElm2.getChildren("str")).iterator();
                        while (rDIT3.hasNext()) {
                            Element intElm3 = (Element) rDIT3.next();
                            String s2 = intElm3.getText();
                            String s3 = intElm3.getAttributeValue("name");
                            if (s3.equals("id")) {
                                mlt_hm.put(s3, s2);
                                mlt_hm.put("mlt_id", s2.replaceAll("\\\\", "\\\\\\\\"));

                            } else {
                                mlt_hm.put(s3, s2);
                            }

                        }
                        Iterator rDIT4 = (intElm2.getChildren("date")).iterator();
                        while (rDIT4.hasNext()) {
                            Element intElm4 = (Element) rDIT4.next();
                            String s2 = intElm4.getText();
                            String s3 = intElm4.getAttributeValue("name");
                            String date = convertToBrief(s2);
                            mlt_hm.put(s3, date);
                        }
                        mlt_list.add(mlt_hm);
                    }
                    res.setMlt_list(mlt_list);
                }

            } else if (rDocument.getAttributeValue("name").equals("facet_counts")) {
                Iterator rDIT = rDocument.getChildren("lst").iterator();

                while (rDIT.hasNext()) {
                    Element facetElm = (Element) rDIT.next();
                    if (facetElm.getAttributeValue("name").equals("facet_fields")) {

                        Iterator ffIT = facetElm.getChildren("lst").iterator();
                        final LinkedHashMap<FilterCatField, List> facetFieldsList = new LinkedHashMap<FilterCatField, List>();

                        final Map facetFieldsLists = new HashMap();

                        while (ffIT.hasNext()) {
                            Element facetFldElm = (Element) ffIT.next();

                            List facetValues = new ArrayList();
                            Iterator fvlIT = facetFldElm.getChildren("int").iterator();
                            /*
                             * code changed here for lpdaac usage of single
                             * chars as names. change is in the length test. Set
                             * to zero for now. jmg
                             */
                            while (fvlIT.hasNext()) {
                                Element facetChildElm = (Element) fvlIT.next();
                                // if
                                // (((facetChildElm.getAttributeValue("name")).trim()).length()
                                // > 1) {
                                if (((facetChildElm.getAttributeValue("name")).trim()).length() > 0) {
                                    Field aField = new Field();
                                    aField.setName(facetChildElm.getAttributeValue("name"));
                                    aField.setEncodedString(URLEncoder.encode(
                                            facetChildElm.getAttributeValue("name"), "UTF-8"));
                                    aField.setValue(facetChildElm.getValue());
                                    aField.setType(facetChildElm.getName());
                                    aField.setDescription(datasourceMapNames.get(facetChildElm
                                            .getAttributeValue("name")));

                                    if (Integer.parseInt(aField.getValue()) != 0) {
                                        facetValues.add(aField);// afield just
                                                                // one row
                                                                // undiff data..
                                    }
                                }
                            }

                            if (!facetValues.isEmpty()) {
                                FilterCatField filtercatField = new FilterCatField();
                                filtercatField.setFilterCatCode(facetFldElm
                                        .getAttributeValue("name"));

                                String localvalue = (String) (filterslist.get(facetFldElm
                                        .getAttributeValue("name")));

                                if ((null != localvalue) && (localvalue.length() > 0)) {
                                    filtercatField.setFilterCatName(localvalue);
                                } else {
                                    filtercatField.setFilterCatName(facetFldElm
                                            .getAttributeValue("name"));
                                }

                                facetFieldsList.put(filtercatField, facetValues);

                            }

                        }
                        res.setFacets(facetFieldsList);

                    }
                }
            }

            else {

            }
        }

        final Element result = root.getChild("result");

        final Integer numFound = new Integer(result.getAttributeValue("numFound"));
        if (!(null == result.getAttributeValue("maxScore"))) {
            final Float maxScore = new Float(result.getAttributeValue("maxScore"));
            res.setMaxScore(maxScore);
        }

        res.setFound(numFound);

        final Integer start = new Integer(result.getAttributeValue("start"));
        res.setStart(start);

        ArrayList al = new ArrayList();
        ArrayList al2 = new ArrayList();
        final Iterator dox = result.getChildren("doc").iterator();

        int count = 0;
        int childctr = 0;

        while (dox.hasNext()) {
            final Element resultDocument = (Element) dox.next();
            final List fieldList = resultDocument.getChildren();
            childctr++;

            final Map<String, Field> docFields = new HashMap<String, Field>();
            for (Iterator iter = fieldList.iterator(); iter.hasNext();) {
                Element triple = (Element) iter.next();
                Field f = new Field();
                f.setName(triple.getAttributeValue("name"));
                f.setValue(triple.getValue());
                List tripleContents = triple.getContent();
                ArrayList extractedValues = new ArrayList();
                if (triple.getName().trim().equalsIgnoreCase("arr")) {
                    for (int k = 0; k < tripleContents.size(); k++) {
                        extractedValues.add(((Element) tripleContents.get(k)).getValue());
                    }
                }
                if (triple.getName().trim().equalsIgnoreCase("str")) {
                    for (int k = 0; k < tripleContents.size(); k++) {
                        extractedValues.add((triple.getTextTrim()));
                    }
                }
                if ((res.getFound() > 0) && (tripleContents.size() == 0)) {
                    extractedValues.add("MetaData Error Encountered" + "  "
                            + triple.getAttributes().get(0).toString());
                }
                f.setType(triple.getName());
                f.setValueList(extractedValues);
                docFields.put(f.getName(), f);
                al.add(f.getValue());
                if (f.getName().equalsIgnoreCase((String) _fields.get(0))) {
                    ArrayList tempValues = f.getValueList();
                    for (Iterator tempiter = tempValues.iterator(); tempiter.hasNext();) {
                        String kvalue = (String) tempiter.next();
                        if (!(al2.contains(kvalue))) {
                            al2.add(kvalue);
                        }
                    }
                }
            }
            Collections.sort(al2);
            res.getDocuments().add(count, docFields);
            res.setJ_Docs(al2);
            count++;
        }
        return res;
    }

    private String getCleanString(String stringToClean) throws IOException {
        final byte xmlBytes[] = stringToClean.getBytes(); // create new byte
                                                          // array

        StringBuffer buffer = new StringBuffer();

        final InputStream is = new ByteArrayInputStream(xmlBytes);// create

        InputStreamReader isr2 = new InputStreamReader(is, "UTF-8"); // Create an

        Reader in2 = new BufferedReader(isr2);// Create a buffering

        int ch;

        while ((ch = in2.read()) > -1) {
            if ((ch == 0x9) || (ch == 0xA) || (ch == 0xD) || ((ch >= 0x20) && (ch <= 0xD7FF))
                    || ((ch >= 0xE000) && (ch <= 0xFFFD)) || ((ch >= 0x10000) && (ch <= 0x10FFFF)))

                buffer.append((char) ch);
        }
        in2.close();

        String myString = buffer.toString();
        return myString;
    }

    /**
     * @param message
     *            Local error message.
     * @param ex
     *            Throwable object caught as exception. Used to send stacktrace.
     * @param params
     *            Solr Parameter list used to build query.
     */
    private void mail(String message, Throwable ex, List<NameValuePair> params) {
        StringBuffer solrInput = new StringBuffer();
        for (NameValuePair parm : params) {
            String name = parm.getName();
            String value = parm.getValue();
            solrInput.append(name + " -> " + value + "\n");
        }

        Writer result = new StringWriter();
        PrintWriter printWriter = new PrintWriter(result);
        ex.printStackTrace(printWriter);

        mail(message + "\n\n" + solrInput.toString() + "\n\n" + result.toString());
    }

    /**
     * @param message
     *            Complete Text error message to embed as body of email.
     */
    private void mail(String message) {

        try {
            EmailGenerator.sendTextMail(toEMAIL, source.toUpperCase() + " Search Error", message,
                    fromEMAIL);
        } catch (Exception exc) {
            exc.printStackTrace();

        }

    }
}
