/**
 * This work was created by participants in the DataONE project, and is
 * jointly copyrighted by participating institutions in DataONE. For 
 * more information on DataONE, see our web site at http://dataone.org.
 *
 *   Copyright ${year}
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and 
 * limitations under the License.
 * 
 * $Id$
 */

package gov.ornl.mercury3.services;

import gov.ornl.mercury3.commands.Tokenz;

import java.io.PrintStream;
import java.io.UnsupportedEncodingException;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.StringTokenizer;

/**
 * Mercury3_EditBoxParser is a class which is intended to allow
 * making changes to a string retrieved from an edit box which might
 * contain one or more quoted selections and one or more
 * boolean operators which might need to be uppercased. Only
 * those operators which are NOT quoted will be changed.
 * 
 * The list of words to be uppercased is embedded. ( currently and,or).
 * 
 * a string of Control characters which need escaping is also embedded.
 * 
 * This change also is ONLY applied to those tokens NOT inside quotes.
 * 
 * 
 * 
 */
public class Mercury3_EditBoxParser {

    private String fSearchText;
    private String facet;
    private static final Set<String> fBOOL_WORDS = new HashSet<String>();
    private static final String fDOUBLE_QUOTE = "\"";

    private static final String fWHITESPACE_AND_QUOTES = " \t\r\n\"";
    private static final String fQUOTES_ONLY = "\"";

    static {
        fBOOL_WORDS.add("or");
        fBOOL_WORDS.add("and");

    }

    /**
     * main method included to allow testing using fixed string.
     * the () which are NOT inside the quotes will be escaped
     * 
     * 
     * Usage:
     * Mercury3_EditBoxParser parser = new Mercury3_EditBoxParser(String);
     * List<Tokenz> ltz = parser.parseSearchText();
     * String fragment = parser.getStringwithBools( ltz);
     * 
     */
    public static void main(String... aArguments) {
        try {
            Mercury3_EditBoxParser parser = new Mercury3_EditBoxParser(
                    "mars and (venus) \"milky or way\" sun \" some obscure (title) and such \" or another ");

            List<Tokenz> ltz = parser.parseSearchText();

            System.out.println(parser.getStringwithBools(ltz));
        } catch (Exception ex) {
            ex.printStackTrace();
        }
    }

    /*
     * getStringwithBools builds a usable string from an ArrayList of Tokenz objects
     * and makes any UN-quoted bools uppercase.
     */
    /** test string :"land cover" or landcover and global fails...zero for nbii from advanced:title
     * test string :("land cover" or landcover) and global returns 6
     * @param ltz
     * @return
     *  
     *   formatted query string for solr
     */
    public String getStringwithBools(List<Tokenz> ltz) {
        String escapeChars = "[\\\\+\\!\\(\\)\\:\\^\\]\\{\\}\\~\\?]";
        Tokenz t1 = new Tokenz();
        String searchr = "";
        String searchr2 = "";
        StringBuffer sb = new StringBuffer();
        boolean b1;
        String aToken = "";
        for (int i = 0; i < ltz.size(); i++) {
            t1 = ltz.get(i);
            String typer = t1.getTokenType();
            b1 = Boolean.parseBoolean(typer);
            aToken = t1.getTokenText().trim();

            if ((aToken.equals("(")) || (aToken.equals(")")) || (aToken.equals("OR"))
                    || (aToken.equals("AND"))) {
                searchr2 = aToken + " ";
            } else if (b1) {
                if (aToken.endsWith(")") || aToken.startsWith("(")) {
                    searchr2 = facet + " : " + aToken + " ";
                } else {
                    searchr2 = facet + " : " + aToken.replaceAll(escapeChars, "\\\\$0") + " ";
                }

                System.out.println(" b1 searchr2 =>" + searchr2);
            } else {
                searchr2 = facet + " : \"" + aToken + "\"";

                System.out.println(" !b1 searchr2 =>" + searchr2);
            }

            sb.append(searchr2);
        }
        return sb.toString();
    }

    /*
     * Constructor which accepts a String object
     */
    public Mercury3_EditBoxParser(String aSearchText) {
        if (aSearchText == null) {
            throw new IllegalArgumentException("Search Text cannot be null.");
        }
        fSearchText = aSearchText;
    }

    /*
     * Constructor which accepts two String object
     */
    public Mercury3_EditBoxParser(String aSearchText, String afacet) {
        if (aSearchText == null) {
            throw new IllegalArgumentException("Search Text cannot be null.");
        }
        fSearchText = aSearchText;//.replaceAll("[(]", " ( ").replaceAll("[)]", " ) ");
        facet = afacet;
    }

    /**
     * Builds an ArrayList of Tokenz objects using 
     * a class scoped String initialized by constructor
     * 
     * 
     *  
     */

    public List<Tokenz> parseSearchText() {
        List<Tokenz> result3 = new ArrayList<Tokenz>();
        boolean returnTokens = true;
        String currentDelims = fWHITESPACE_AND_QUOTES;
        StringTokenizer parser = new StringTokenizer(fSearchText, currentDelims, returnTokens);
        String token = null;
        while (parser.hasMoreTokens()) {
            token = parser.nextToken(currentDelims);
            if (!isDoubleQuote(token)) {
                //addTokensToResult(token, result3, currentDelims);
                addTokensToResult(token, result3, currentDelims, facet);
            } else {
                currentDelims = flipDelimiters(currentDelims);
            }
        }
        return result3;
    }

    public List<Tokenz> parseSearchText(String querystring) {
        List<Tokenz> result3 = new ArrayList<Tokenz>();
        boolean returnTokens = true;
        String currentDelims = fWHITESPACE_AND_QUOTES;
        StringTokenizer parser = new StringTokenizer(querystring, currentDelims, returnTokens);
        String token = null;
        while (parser.hasMoreTokens()) {
            token = parser.nextToken(currentDelims);
            if (!isDoubleQuote(token)) {
                //addTokensToResult(token, result3, currentDelims);
                addTokensToResult(token, result3, currentDelims, facet);
            } else {
                currentDelims = flipDelimiters(currentDelims);
            }
        }
        return result3;
    }

    public List<Tokenz> parseSearchText_adv() {
        boolean b1 = false;
        StringBuffer sb = new StringBuffer();
        String typer = null;
        String nu_qryStr = null;
        List<Tokenz> result3 = new ArrayList<Tokenz>();
        boolean returnTokens = true;
        String currentDelims = fWHITESPACE_AND_QUOTES;
        StringTokenizer parser = new StringTokenizer(fSearchText, currentDelims, returnTokens);
        String token = null;
        while (parser.hasMoreTokens()) {
            token = parser.nextToken(currentDelims);
            if (!isDoubleQuote(token)) {
                //addTokensToResult(token, result3, currentDelims);
                addTokensToResult(token, result3, currentDelims, facet);
            } else {
                currentDelims = flipDelimiters(currentDelims);
            }
        }

        for (Tokenz tk : result3) {
            typer = tk.getTokenType();
            b1 = Boolean.parseBoolean(typer);
            if (!b1) {
                sb.append(" \" " + tk.getTokenText() + " \" ");
            } else {
                sb.append(" " + tk.getTokenText().replace("(", " ( ").replace(")", " ) "));
            }
            nu_qryStr = sb.toString();
        }

        PrintStream out8;
        try {
            out8 = new PrintStream(System.out, true, "UTF-8");
            out8.println("old fSearchText => " + fSearchText + "\n");
            out8.println("nu_qryStr => " + nu_qryStr + "\n");
        } catch (UnsupportedEncodingException e) {
            // TODO Auto-generated catch block
            e.printStackTrace();
        }

        List<Tokenz> result4 = parseSearchText(nu_qryStr);

        return result4;
    }

    private boolean isBoolWord(String aSearchTokenCandidate) {
        return fBOOL_WORDS.contains(aSearchTokenCandidate);
    }

    private boolean textHasContent(String aText) {
        return (aText != null) && (!aText.trim().equals(""));
    }

    /*
     * 
     */
    private void addTokensToResult(String aToken, List<Tokenz> aResult3, String currentDelims) {
        String type = "";
        if (textHasContent(aToken) && !isBoolWord(aToken.trim())) {
            type = Boolean.toString(!currentDelims.equals(fDOUBLE_QUOTE));
            aResult3.add(new Tokenz(type, aToken.trim()));// aToken.trim() );
        } else if (textHasContent(aToken) && isBoolWord(aToken.trim())) {
            type = Boolean.toString(!currentDelims.equals(fDOUBLE_QUOTE));
            aResult3.add(new Tokenz(type, aToken.trim().toUpperCase()));// aToken.trim() );
        }
    }

    private void addTokensToResult(String aToken, List<Tokenz> aResult3, String currentDelims,
            String facet) {
        String type = "";
        if (textHasContent(aToken) && !isBoolWord(aToken.trim())) {
            type = Boolean.toString(!currentDelims.equals(fDOUBLE_QUOTE));

            aResult3.add(new Tokenz(type, aToken.trim()));

        } else if (textHasContent(aToken) && isBoolWord(aToken.trim())) {
            type = Boolean.toString(!currentDelims.equals(fDOUBLE_QUOTE));
            aResult3.add(new Tokenz(type, aToken.trim().toUpperCase()));// aToken.trim() );
        }
    }

    private boolean isDoubleQuote(String aToken) {
        return aToken.equals(fDOUBLE_QUOTE);
    }

    /*
     * Changes the defined delimiter to allow us to keep quoted strings as a single token
     */
    private String flipDelimiters(String aCurrentDelims) {
        String result = null;
        if (aCurrentDelims.equals(fWHITESPACE_AND_QUOTES)) {
            result = fQUOTES_ONLY;
        } else {
            result = fWHITESPACE_AND_QUOTES;
        }
        return result;
    }
}
