% The DataProcessor class processes ASCII time series data for a given interval
% and produces plots of the data.  It currently handles the processing of ASCII 
% data produced by WetLabs FLNTUs, Seabird SBE37s, Seabird SBE16 CTDs.
% This class relies on the edu.hawaii.soest.bbl.configuration.Configure 
% class to get runtime configuration information.
  
%  Copyright: 2007 Regents of the University of Hawaii and the
%             School of Ocean and Earth Science and Technology
% 
%    Purpose: To process ASCII data in multiple increments (daily, weekly, etc)
%             and produce plots of observations.
%    Authors: Christopher Jones
%             Judith Wells
%             Kristen Fogaren
%    History: This code was adapted from code written by Eufemia Palomino at 
%             the University of California Santa Cruz for the Network for 
%             Environmental Observations of the Coastal Ocean (NEOCO) project
%             in 2003, and by FLNTU processing code originally written by 
%             Judith Wells and Kristen Fogaren at the University of Hawaii at
%             Manoa.
%  
% $HeadURL: https://bbl.ancl.hawaii.edu/projects/bbl/trunk/src/matlab/+edu/+hawaii/+soest/+bbl/+processing/DataProcessor.m $
% $LastChangedDate: 2014-07-16 21:51:16 -0600 (Wed, 16 Jul 2014) $
% $LastChangedBy: kilonalu $
% $LastChangedRevision: 1134 $
% 
% This program is free software; you can redistribute it and/or modify
% it under the terms of the GNU General Public License as published by
% the Free Software Foundation; either version 2 of the License, or
% (at your option) any later version.
% 
% This program is distributed in the hope that it will be useful,
% but WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
% GNU General Public License for more details.
% 
% You should have received a copy of the GNU General Public License
% along with this program; if not, write to the Free Software
% Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
% 
classdef DataProcessor < hgsetget & dynamicprops

  properties % of this class
  
    % The instance of the Configure class used to provide configuration
    % details for this DataProcessor
    configuration;
    
    % The parsed ASCII string of the data as a cell array of observations
    dataCellArray;
    
    % The raw ASCII string of data from the RBNB Data Turbine to be processed
    dataString;
    
    % The times associated with each frame of data fetched from the RBNB
    dataTimes;
    
    % The name returned by the RBNB Data Turbine
    dataName;
    
    % The start time of the timer object
    timerStartTime;
    
    % The timer interval for scheduled processing in minutes
    timerInterval = 20;
    
    % The timer object used to schedule processing
    timerObject;
    
    % The time of the current processing
    processTime;
  end % properties
  
  methods % functions available from this class
    
    % The Constructor: creates an instance of the DataProcessor class
    % @returns dataProcessor - an instance of the DataProcessor
    function self = DataProcessor(configuration)
      % set the configuration information for this processing instance
      self.configuration  = configuration;
    end % DataProcessor
    
    % A method used to parse the raw data string. This method assumes that
    % the DataProcessor.configuration.dataFormatString is set to provide data
    % typing information, Dataprocessor.configuration.duration is set to 
    % provide relative size information, Dataprocessor.configuration.fieldDelimiter
    % is set to provide the field delimiter character, and that 
    % Dataprocessor.configuration.numberOfHeaderLines is set to provide header 
    % information for the raw data string.
    % @returns dataCellArray
    function dataCellArray = parse(self)
      
      if ( self.configuration.debug )
        disp('DataProcessor.parse() called.');
      end
      
      dataLinesCellArray = strread(self.dataString, '%s',             ...
                                   'delimiter', self.configuration.recordDelimiter);
      
      % Evaluate each data line for incorrect parsing.  If it fails to parse,
      % set the line to the empty string as QA/QC
      for lineNumber = 1:length(dataLinesCellArray)
        
        if ( ~isempty(dataLinesCellArray{lineNumber}) ) 
            try
              textscan(                                               ...
                dataLinesCellArray{lineNumber},                       ...
                self.configuration.dataFormatString,                  ...
                'ReturnOnError', 0,                                   ...
                'Delimiter', self.configuration.fieldDelimiter,       ...
                'MultipleDelimsAsOne', self.configuration.ignoreMultipleDelims ...
              );
              
              % add the line ending that was stripped off
              dataLinesCellArray{lineNumber} = ...
                [dataLinesCellArray{lineNumber} char(13) char(10)];
            
            catch parseException 
              disp(['There was a problem parsing line ' num2str(lineNumber) ...
                    '. It has been discarded. The line was:']               ...
                  );
              disp(dataLinesCellArray{lineNumber});
              dataLinesCellArray{lineNumber} = '';
              disp(['The error was: ' parseException.message]);
              
            end
            
        end
        
      end

      % transform the array in case cells are irregular length
      invertedDataLinesCellArray = {dataLinesCellArray{:}};
      % convert the cell array of data lines back to a character array
      dataLinesArray = cell2mat(invertedDataLinesCellArray);
      
      % now that the data string has been QA/QC'd, produce a cell array
      % via textscan
      dataCellArray =                                           ...
        textscan(                                               ...
          dataLinesArray',                                      ...
          self.configuration.dataFormatString,                  ...
          'BufSize', (10 * self.configuration.duration),        ...
          'Delimiter', self.configuration.fieldDelimiter,       ...
          'MultipleDelimsAsOne', self.configuration.ignoreMultipleDelims, ...
          'headerLines', 0 ...
        );
      
    end %parse
    
    % A method used to process the raw data string, build derived variables,
    % create figures for the designated variables, and export the figures
    % to the designated formats.
    % @returns void
    function process(self)
      
      if ( self.configuration.debug )
        disp('DataProcessor.process() called.');
      end      
      set(self, 'processTime', now());
      
      %Check if importing all data from Data Start Date
      if strcmp('full',self.configuration.duration) || ...
              strcmp('full',self.configuration.duration_days)
          self.configuration.duration_days = ...
              round(now - datenum(self.configuration.dataStartDate));
      end
      
      %Check dataEndDate
      if strcmp(self.configuration.dataEndDate, 'now')
          self.configuration.dataEndDate = ' ';
      end
      
      %Match import duration in seconds with import duration in days
      if self.configuration.duration_days ~= 0
          self.configuration.duration = self.configuration.duration_days*24*3600;
      else
          self.configuration.duration_days = self.configuration.duration/3600/24;
      end
      
      % get the most recent interval of data
      if strcmp(self.configuration.rbnbServer,'LOBOViz')
        [self.dataString, ...
         self.dataTimes,  ...
         self.dataName] = self.getLOBOVizData();
      else if ( self.configuration.readArchive )
             [self.dataString, ...
              self.dataTimes,  ...
              self.dataName] = self.getArchiveData();
          else
             [self.dataString, ...
              self.dataTimes,  ...
              self.dataName] = self.getRBNBData();
          end
      end
      
      % parse the data string into a cell array used to create graphics
      self.dataCellArray = self.parse();
      
      % Create derived variables.  For each variable name in the list of derived
      % variables, calculate the derived variable and append it to the
      % dataCellArray for later use.  Also update the dataVariableNames and
      % dataVariableUnits properties.            
      try
        for derivedVariableNumber = 1:length(self.configuration.derivedVariableNames)
          self.dataCellArray{length(self.dataCellArray) + 1} = ...
            createDerivedVariable(self, ...
              self.configuration.derivedVariableNames{ ...
                derivedVariableNumber ...
              } ...
            );
        end
      catch derivedVariableException
        disp(derivedVariableException.message);
        return;
      end
      
      % get the time vector
        time = self.dataCellArray{find(strcmp ...
                 ('serialdate',self.configuration.dataVariableNames))};
             
      % create a depth column adjusted to MLLW (if applicable)
      if any(strcmp('depth',self.configuration.dataVariableNames))
          self.dataCellArray{length(self.dataCellArray) + 1} = ...
              depthAdjustment(self,time);
      end
             
      % remove data prior to the Data Start Date
      try
       dataStartDate = datenum(self.configuration.dataStartDate,        ...
         'mm-dd-yyyy HH:MM:SS');
       if time(1) < dataStartDate
          index=find(time>dataStartDate);
          time=time(index);
          for i=1:length(self.dataCellArray)
              self.dataCellArray{i}=self.dataCellArray{i}(index);
          end
          clear i index
       end
      catch dataStartDateException
          disp(dataStartDateException.message);
      end
      
      %Remove data following the Data End Date
      if ~strcmp(self.configuration.dataEndDate,' ')
          try 
          dataEndDate = datenum(self.configuration.dataEndDate,         ...
              'mm-dd-yyyy HH:MM:SS');
       if time(end) > dataEndDate
          index=find(time<dataEndDate);
          time=time(index);
          for i=1:length(self.dataCellArray)
              self.dataCellArray{i}=self.dataCellArray{i}(index);
          end
          clear i index
       end
      catch dataEndDateException
          disp(dataEndDateException.message);
          end
      end
      
          
      % create figures as outlined in the configuration properties
      if ( self.configuration.createFigures )
        if ( strcmp(self.configuration.currentFigureType, 'timeSeries') )
          figurePropertiesArray = self.configuration.timeSeriesFigures;
         
        elseif ( strcmp(self.configuration.currentFigureType, 'temperatureSalinity') )
          figurePropertiesArray = self.configuration.tsFigures;
        else
          disp(['The figure type is not recognized.  Please set the ' ...
                'currentFigureType to either timeSeries or '          ...
                'temperatureSalinity.']);
        end

        % define the base output directory
        outdirectory=self.configuration.outputDirectory;
            
        % set the output format(s)
        outputFormat=self.configuration.outputFormat;
            
        for figureNumber = 1:length(figurePropertiesArray)
          try
            % Get the figure title prefix string
            figureTitlePrefix = ...
            char( ...
              figurePropertiesArray{figureNumber}(1) ...
            );
            
            % Get the figure start date in UTC time
            figureStartDate = ...
              char( ...
                figurePropertiesArray{figureNumber}(2) ...             
              );
            
            % Get the figure duration in seconds
            figureDuration = ...
              str2num( ...
                char( ...
                  figurePropertiesArray{figureNumber}(3) ...             
                ) ...
              );
            
            % Get the figure Y axis cell array
            figureYAxisVariables = ...
              figurePropertiesArray{figureNumber}(4);
            
            % Get the figure X axis cell array
            figureXAxisVariables = ...
              figurePropertiesArray{figureNumber}(5);
            
            % Get the figure X axis tick step
            figureXAxisTickStep = ...
              str2num( ...
                char( ...
                figurePropertiesArray{figureNumber}(6) ...
                ) ...
              );
            
            % Get the figure tick formats for each plot
            plotTickFormats = ...
                  figurePropertiesArray{figureNumber}(7);
            
            % Get the graphic markers for each plot
            graphicMarkers = ...
              figurePropertiesArray{figureNumber}(8);
            
            % Get the graphic marker sizes for each plot
            graphicMarkersizes = ...
              figurePropertiesArray{figureNumber}(9);
            
            % Get the graphic markers colors for each plot
            graphicMarkerColors = ...
              figurePropertiesArray{figureNumber}(10);
            
            % Get the includeMovingAverage boolean value
            includeMovingAverage = ...
              str2num( ...
                char( ...
                  figurePropertiesArray{figureNumber}(11) ...
                ) ...
              );
              
            % Get the moving average duration value
            movingAverageDuration = ...
              str2num( ...
                char( ...
                  figurePropertiesArray{figureNumber}(12) ...
                ) ...
              );

            % Get the moving average line color
            movingAverageLineColor = ...
              figurePropertiesArray{figureNumber}(13);
            
            % Get the moving average line width
            movingAverageLineWidth = ...
              str2num( ...
                char( ...exp
                  figurePropertiesArray{figureNumber}(14) ...
                ) ...
              );
            
            % now create the figures
            
            if ( strcmp(self.configuration.currentFigureType, 'timeSeries') )
            figureHandle =                  ...
              self.createTimesSeriesFigure( ...
                figureTitlePrefix         , ...
                figureStartDate           , ...
                figureDuration            , ...
                figureYAxisVariables      , ...
                figureXAxisVariables      , ...
                figureXAxisTickStep       , ...
                plotTickFormats           , ...
                graphicMarkers            , ...
                graphicMarkersizes        , ...
                graphicMarkerColors       , ...
                includeMovingAverage      , ...
                movingAverageDuration     , ...
                movingAverageLineColor    , ...
                movingAverageLineWidth      ...
              );
              
            elseif ( strcmp(self.configuration.currentFigureType, 'temperatureSalinity') )
              figureHandle =                  ...
                self.createTSFigure(          ...
                  figureTitlePrefix         , ...
                  figureStartDate           , ...
                  figureDuration            , ...
                  figureYAxisVariables      , ...
                  figureXAxisVariables      , ...
                  figureXAxisTickStep       , ...
                  plotTickFormats           , ...
                  graphicMarkers            , ...
                  graphicMarkersizes        , ...
                  graphicMarkerColors       , ...
                  includeMovingAverage      , ...
                  movingAverageDuration     , ...
                  movingAverageLineColor    , ...
                  movingAverageLineWidth      ...
                );
                           
            end
            
            % export figures as outlined in the configuration properties
            % Todo: Needs work to make the export take format arguments
            %if ( self.configuration.exportFigures )
              % call the export method
             % outputFormat = '';
             % figureNameSuffix = [num2str(figureDuration/60/60/24) 'day'];
             % self.export(figureHandle, outputFormat, figureNameSuffix);
              % self.export(TSHandle);
            %end
           
            % export figures as outlined in the configuration properties
                if ( self.configuration.exportFigures )
                    % call the export method
                    figureNameSuffix = [num2str(figureDuration/60/60/24) 'day'];
                    self.export2(                 ...
                        figureHandle            , ...
                        time                    , ...
                        outputFormat            , ...
                        outdirectory            , ...
                        figureNameSuffix          ...
                                                        );
                end            
            
            % clean up variables
            clear figureTitlePrefix    ...
                  figureDuration       ...
                  figureYAxisVariables ...
                  figureXAxisVariables ...
                  figureXAxisTickStep  ...
                  figureHandle;
            
          catch figureException
            disp(figureException.message);
          end % end try statement
        end % end for loop
                
      end % end if statement
      
      % create figures for PacIOOS website
        if (self.configuration.createPacIOOSFigures)
                
            % define the base output directory
            outDir = [self.configuration.outputDirectory 'PacIOOSplots'  ...
                      self.configuration.pathSeparator];           
            
            % get the figure graphing properties
            figureProperties = self.configuration.PacIOOSFigureProperties;
            
            % get the variable names
                yVariables = figureProperties{1};
                
                % get the axis labels
                yAxisLabels = figureProperties{2};
                
                % get the plot locations
                plotLocations = cell2mat(figureProperties{3});
                
                % get the plot colors
                plotColors = figureProperties{4};
                
                % get the range info for the y-axis
                variableRanges = figureProperties{5};
                
            
            % get the figure info (title prefix & duration)
            figureInfo=self.configuration.PacIOOSFigures;

            % create the figures
            for plotNumber=1:length(figureInfo)
                
                % get the figure title prefix string
                figureTitlePrefix = char(figureInfo{plotNumber}{1});

                % get the figure duration
                figureDuration = char(figureInfo{plotNumber}{2});
                
                % get the output format
                outputFormat = char(figureInfo{plotNumber}{3});
                
                %check if making multiple monthly plots
                if strcmp(figureDuration,'monthly') &&                  ...
                        self.configuration.duration_days > 32
                  DCA=self.dataCellArray;    
                  yearSpan = (str2double(datestr(time(end),'yyyy'))     ...
                            : -1 : str2double(datestr(time(1),'yyyy')));
                  if length(yearSpan) > 1         
                     monthSpan = (12 : -1 : 1);
                  else
                      monthSpan = (str2double(datestr(time(end),'mm'))  ...
                            : -1 : str2double(datestr(time(1),'mm')));
                  end
                  
                  for yr = 1:length(yearSpan)
                      for mo=1:length(monthSpan)
                          startDay=datenum(yearSpan(yr),monthSpan(mo),1);
                          if mo ~= 12
                              endDay=datenum(yearSpan(yr),monthSpan(mo)+1,1);
                          else
                              endDay=datenum(yearSpan(yr)+1,1,1);
                          end
                          if startDay <= time(1)-1
                              break
                          else
                              %Pull out individual months of data
                              index=find(time>startDay & time<endDay);
                              for i=1:length(self.dataCellArray)
                                 self.dataCellArray{i}=DCA{i}(index);
                              end
                                 
                                  % call the plotting function
                                  figureHandle =                 ...
                                    self.createPacIOOSFigures(   ...
                                     time(index)               , ...
                                     figureTitlePrefix         , ...                      
                                     figureDuration            , ...
                                     yVariables                , ...
                                     yAxisLabels               , ...
                                     plotLocations             , ...
                                     plotColors                , ...
                                     variableRanges              ...
                                                                      );

                               if ( self.configuration.exportFigures )
                                 % call the export method
                                  figureNameSuffix = datestr(startDay,'yyyy_mm');
                                  
                                     self.exportMonthly(           ...
                                         figureHandle            , ...
                                         time(index)             , ...
                                         outputFormat            , ...
                                         outDir                  , ...
                                         figureNameSuffix          ...
                                                                )        ;
                               end
                          end
                          clear i index
                      end  %month loop
                  end    %year loop
                  
                  self.dataCellArray = DCA;
                 %clear the loop variables
                  clear yearSpan            ...
                        monthSpan           ...
                        yr                  ...
                        mo                  ...
                        startDay            ...
                        endDay              ...
                        DCA                 
                  
                else
                                             
                 % call the plotting function
                 figureHandle =                   ...
                    self.createPacIOOSFigures(    ...
                      time                      , ...
                      figureTitlePrefix         , ...                      
                      figureDuration            , ...
                      yVariables                , ...
                      yAxisLabels               , ...
                      plotLocations             , ...
                      plotColors                , ...
                      variableRanges              ...
                                                        );

                 if ( self.configuration.exportFigures )
                    % call the export method
                    if strcmp(figureDuration,'monthly')
                      if strcmp(datestr(time(end),'yyyy_mm'), ...
                                 datestr(now,'yyyy_mm')) || ...
                         strcmp(datestr(time(end)+1,'yyyy_mm'), ...
                                 datestr(now,'yyyy_mm'))
                        figureNameSuffix = datestr(time(end),'yyyy_mm');
                        self.exportMonthly(           ...
                            figureHandle            , ...
                            time                    , ...
                            outputFormat            , ...
                            outDir                  , ...
                            figureNameSuffix          ...
                                           )        ;
                      end
                    else
                        figureNameSuffix = [figureDuration 'day'];                    
                        self.export2(                 ...
                            figureHandle            , ...
                            time                    , ...
                            outputFormat            , ...
                            outDir                  , ...
                            figureNameSuffix          ...
                                     )              ;
                    end                                   
                 end
                end

                % clean up variables
                clear figureTitlePrefix    ...
                      figureDuration       ...
                      figureHandle         ...
                      figureNameSuffix
              
            end % end for loop
                  
        end % end if statement      
      
      if ( self.configuration.debug )
        disp(['Processing complete.  Next process time: ' ...
              datestr(self.processTime + self.configuration.timerInterval/60/24, ...
                      'mm-dd-yyyy HH:MM')]);
      end
      
      if self.configuration.closePlotWindow
          close
      end
      
    end %process
    
    % A method used to produce derived variables from the raw data.  This 
    % function can currently handle the following derivations:
    % serialdate: from date & time, or from datetime
    % depth: from pressure
    % turbidity: from turbidityVolts
    % chlorophyll: from chlorophyllVolts
    % The variable returned has the same length as the dataCellArray so it
    % can be appended to that cell array for later use.
    % @returns value - the derived variable array
    function value = createDerivedVariable(self, derivedVariableName)
      
      if ( self.configuration.debug )
        disp('DataProcessor.createDerivedVariable() called.');
      end

      % derive each of the named variables
      switch derivedVariableName
        
        % derive the serial date from either the date & time, or datetime
        case self.configuration.serialdateFieldName
          if ( ~isempty(self.dataCellArray) ) 
            % Extract timestamp information out of the data cell array, and append a
            % serial date vector to the end of the data cell array for use in plotting.

            % Find which data columns represents the date & time columns 
            % or just the datetime column, and produce a vector of serial dates.  
            % Test that the variable units are also present in order to interpret 
            % the date, time, or datetimes.
            if ( any(strcmp(self.configuration.dateFieldName, ...
                                      self.configuration.dataVariableNames)) && ...
                 any(strcmp(self.configuration.timeFieldName, ...
                                      self.configuration.dataVariableNames)) && ...
                 length(self.configuration.dataVariableNames) == ...
                 length(self.configuration.dataVariableUnits) )    

              % use datenum(X, format), and paste the vectors together:
              % ([datevector spacevector timevector])
              % the find(strcmp()) is used to find the indices of the date and time
              % columns, using the configuration fields that designate which column
              % represents the date, time, or datetime columns
              value = datenum( ...
                        [char(self.dataCellArray{ ...
                           find( ...
                             strcmp( ...
                               self.configuration.dateFieldName, ...
                               self.configuration.dataVariableNames ...
                             ) ...
                           ) ...
                         }) ...
                         repmat(' ', ...
                           length( ...
                             self.dataCellArray{ ...
                               find( ...
                                 strcmp( ...
                                   self.configuration.dateFieldName, ...
                                   self.configuration.dataVariableNames ...
                                 ) ...
                               ) ...
                             } ...
                           ), 1 ...
                         ) ...
                         char(self.dataCellArray{ ...
                           find( ...
                             strcmp( ...
                               self.configuration.timeFieldName, ...
                               self.configuration.dataVariableNames ...
                             ) ...
                           ) ...
                         }) ...
                        ], ...
                        [self.configuration.dataVariableUnits{ ...
                          find(strcmp( ...
                            self.configuration.dateFieldName, ...
                            self.configuration.dataVariableNames) ...
                          ) ...
                         } ...
                         ' ' ...
                         self.configuration.dataVariableUnits{ ...
                           find( ...
                             strcmp( ...
                               self.configuration.timeFieldName, ...
                               self.configuration.dataVariableNames) ...
                           ) ...
                         } ...
                        ] ...
                      );
                      
              % update the field names and units arrays
              updateDataVariableNames(self, self.configuration.serialdateFieldName);
              updateDataVariableUnits(self, 'days');
                      
            % the dates and times are in one column, datetime
            elseif ( any(strcmp(self.configuration.datetimeFieldName, ...
                                          self.configuration.dataVariableNames))  && ...
                                  length(self.configuration.dataVariableNames) == ...
                                  length(self.configuration.dataVariableUnits) ) 

              % use datenum(X, format)
              value = datenum( ...
                        char(self.dataCellArray{ ...
                           find( ...
                             strcmp( ...
                               self.configuration.datetimeFieldName, ...
                               self.configuration.dataVariableNames ...
                             ) ...
                           ) ...
                         }), ...
                        self.configuration.dataVariableUnits{ ...
                          find(strcmp( ...
                            self.configuration.datetimeFieldName, ...
                            self.configuration.dataVariableNames) ...
                          ) ...
                         } ...
                      );
              % update the field names and units arrays
              updateDataVariableNames(self, self.configuration.serialdateFieldName);
              updateDataVariableUnits(self, 'days');
              
            else
              value = [];
              error(['There are no date, time, or datetime fields designated ' ...
                     'in the dataVariableNames cell array.  Failed to build ' ...
                     'serial date vector. ' ...
                     'Set the dataVariableNames property and the dataVariableUnits ' ...
                     'property to include "date" and "time" names or a "datetime" name ' ...
                     'and their respective format strings.']);
            end % end if statement
          else
            % we have no data yet, return an empty array
            value = [];
          end % end if statement (~isempty(self.dataCellArray))
        
        % derive the depth from pressure  
        case self.configuration.depthFieldName
          if ( ~isempty(self.dataCellArray) ) 
            
            % 3. calculate the gravity variation with latitude and pressure:
            % 21.16 = latitude of Honolulu
            pressure = self.dataCellArray{ ...
              find( ...
                strcmp( ...
                  self.configuration.pressureFieldName, ...
                  self.configuration.dataVariableNames ...
                ) ...
              ) ...
            };
            
            x  = [sin(self.configuration.sensorLatitude/57.29578)]^2;
            g1 = 9.780318 .* ...
                 [1 + (5.2788*10^-3 + 2.36*10^-5 .* x) .* x] ...
                 + 1.092*10^-6 .* pressure;

            % 4. calculate depth using g
            depth = [...
              ( ...
                ( ...
                  (-1.82*10^-15 .* pressure + 2.279*10^-10) .* ...
                  pressure - 2.2512*10^-5 ...
                ) .* pressure + 9.72659 ...
              ) .* pressure ...
            ] ./ g1;
            
            value = depth;
            
            clear x g1 pressure depth;            
            % update the field names and units arrays
            updateDataVariableNames(self, self.configuration.depthFieldName);
            updateDataVariableUnits(self, 'm');

            
          else
            value = [];
          end % end if statement (~isempty(self.dataCellArray))
        
        % derive turbidity from turbidity volts
        case self.configuration.turbidityFieldName
          if ( ~isempty(self.dataCellArray) ) 
            if ( any(strcmp(self.configuration.turbidityVoltageFieldName, ...
                                      self.configuration.dataVariableNames)) && ...
                 length(self.configuration.dataVariableNames) == ...
                 length(self.configuration.dataVariableUnits) )    
              
              % constants from the WetLabs FLNTU device file              
              
              % get a reference to the turbidity dark counts
              darkCounts = ...
                self.configuration.calibrationCoefficientValues{ ...
                  find( ...
                    strcmp( ...
                      self.configuration.turbidityDarkCountsFieldName, ...
                      self.configuration.calibrationCoefficientNames   ...
                    ) ...
                  ) ...
                };

              % get a reference to the turbidity scale factor
              scaleFactor = ...
                self.configuration.calibrationCoefficientValues{ ...
                  find( ...
                    strcmp( ...
                      self.configuration.turbidityScaleFactorFieldName, ...
                      self.configuration.calibrationCoefficientNames   ...
                    ) ...
                  ) ...
                };

              % get a reference to the turbidity volts array
              turbidityVoltsArray = ...
                self.dataCellArray{ ...
                  find( ...
                    strcmp( ...
                      self.configuration.turbidityVoltageFieldName, ...
                      self.configuration.dataVariableNames ...
                    ) ...
                  ) ...
                };
              
              % build an array of ones for set-based multiplication
              onesArray     = ...
                ones(length(self.dataCellArray{ ...
                              find( ...
                                strcmp( ...
                                  self.configuration.turbidityVoltageFieldName, ...
                                  self.configuration.dataVariableNames ...
                                ) ...
                              ) ...
                            } ...
                     ), 1 ...
                );
              
              darkCountsArray = onesArray .* darkCounts;              
              scaleFactorArray = onesArray .* scaleFactor;
              
              % NTU = Scale Factor x (Output - Dark Counts)
              value = scaleFactorArray .* (turbidityVoltsArray - darkCountsArray);
              
              clear darkCounts scaleFactor maximumOutput resolution ...
                    darkCountsArray scaleFactorArray onesArray;
              
              % update the field names and units arrays
              updateDataVariableNames(self, self.configuration.turbidityFieldName);
              updateDataVariableUnits(self, 'NTU');

            else
              error(['There is no turbidityVolts field designated ' ...
                     'in the dataVariableNames cell array.  Failed to build ' ...
                     'turbidity vector. ' ...
                     'Set the dataVariableNames property and the dataVariableUnits ' ...
                     'property to include "turbidityVolts" name ' ...
                     'and their respective format strings.']);              
            end
            
          else
            value = [];
          end % end if statement (~isempty(self.dataCellArray))
        
        % derive chlorophyll (ug/L) from chlorophyll volts (or from RFU values)
        case self.configuration.chlorophyllFieldName
          if ( ~isempty(self.dataCellArray) )
              
            if ( any(strcmp(self.configuration.chlorophyllRFUFieldName, ...
                                      self.configuration.dataVariableNames)) && ...
                 length(self.configuration.dataVariableNames) == ...
                 length(self.configuration.dataVariableUnits) )
             
             % get a reference to the chlorophyll RFU array
              chlorophyllRFUArray = ...
                self.dataCellArray{ ...
                  find( ...
                    strcmp( ...
                      self.configuration.chlorophyllRFUFieldName, ...
                      self.configuration.dataVariableNames ...
                    ) ...
                  ) ...
                };
            
              value = chlorophyllRFUArray * 4.55 - 0.88                     ;
              
              % update the field names and units arrays
              updateDataVariableNames(self, self.configuration.chlorophyllFieldName);
              updateDataVariableUnits(self, '\mug/l');
             
                          
            else if ( any(strcmp(self.configuration.chlorophyllVoltageFieldName, ...
                                      self.configuration.dataVariableNames)) && ...
                 length(self.configuration.dataVariableNames) == ...
                 length(self.configuration.dataVariableUnits) )    
              
              % constants from the WetLabs FLNTU device file              
              
              % get a reference to the chlorophyll dark counts
              darkCounts = ...
                self.configuration.calibrationCoefficientValues{ ...
                  find( ...
                    strcmp( ...
                      self.configuration.chlorophyllDarkCountsFieldName, ...
                      self.configuration.calibrationCoefficientNames     ...
                    ) ...
                  ) ...
                };
              
              % get a reference to the chlorophyll scale factor
              scaleFactor = ...
                self.configuration.calibrationCoefficientValues{ ...
                  find( ...
                    strcmp( ...
                      self.configuration.chlorophyllScaleFactorFieldName, ...
                      self.configuration.calibrationCoefficientNames      ...
                    ) ...
                  ) ...
                };
               
              % get a reference to the chlorophyll volts array
              chlorophyllVoltsArray = ...
                self.dataCellArray{ ...
                  find( ...
                    strcmp( ...
                      self.configuration.chlorophyllVoltageFieldName, ...
                      self.configuration.dataVariableNames ...
                    ) ...
                  ) ...
                };
              
              % build an array of ones for set-based multiplication
              onesArray     = ...
                ones(length(self.dataCellArray{ ...
                              find( ...
                                strcmp( ...
                                  self.configuration.chlorophyllVoltageFieldName, ...
                                  self.configuration.dataVariableNames ...
                                ) ...
                              ) ...
                            } ...
                     ), 1 ...
                );
              
              darkCountsArray = onesArray .* darkCounts;              
              scaleFactorArray = onesArray .* scaleFactor;
              
              % CHL (µg/l) = Scale Factor x (Output - Dark Counts)
              value = scaleFactorArray .* (chlorophyllVoltsArray - darkCountsArray);
              
              clear darkCounts scaleFactor maximumOutput resolution ...
                    darkCountsArray scaleFactorArray onesArray;
         
              % update the field names and units arrays
              updateDataVariableNames(self, self.configuration.chlorophyllFieldName);
              updateDataVariableUnits(self, '\mug/l');
            
            else
              error(['There is no chlorophyllVolts field designated ' ...
                     'in the dataVariableNames cell array.  Failed to build ' ...
                     'chlorophyll vector. ' ...
                     'Set the dataVariableNames property and the dataVariableUnits ' ...
                     'property to include "chlorophyllVolts" name ' ...
                     'and their respective format strings.']);
                end
            end
          else
            value = [];
          end % end if statement (~isempty(self.dataCellArray))
                    
        % derive dissolved oxygen (uM) from dissolved oxygen phase
        % (or from mg/L)
        case self.configuration.dissolvedOxygenFieldName
          if ( ~isempty(self.dataCellArray) )              

            if ( any(strcmp(self.configuration.dissolvedOxygenMetricFieldName,   ...
                                      self.configuration.dataVariableNames)) && ...                    
                 length(self.configuration.dataVariableNames) ==                ...
                 length(self.configuration.dataVariableUnits) )
             
            % get a reference to the dissolved oxygen array
              dissolvedOxygenMetricArray = ...
                self.dataCellArray{ ...
                  find( ...
                    strcmp( ...
                      self.configuration.dissolvedOxygenMetricFieldName, ...
                      self.configuration.dataVariableNames ...
                    ) ...
                  ) ...
                };
            
            % get a reference to the units for dissolved oxygen
              dissolvedOxygenUnits = ...
                self.configuration.dataVariableUnits{ ...
                  find( ...
                    strcmp( ...
                      self.configuration.dissolvedOxygenMetricFieldName, ...
                      self.configuration.dataVariableNames ...
                    ) ...
                  ) ...
                };
            
            % convert from mg/L or mL/L to micro Molar
              if strcmp(dissolvedOxygenUnits,'mg/L') 
                value = dissolvedOxygenMetricArray * 35.2152;
              else if strcmp(dissolvedOxygenUnits,'mL/L')
                      value = dissolvedOxygenMetricArray * 44.9826;
                  else
                      value = dissolvedOxygenMetricArray;
                  end
              end
            
            % update the field names and units arrays
              updateDataVariableNames(self, self.configuration.dissolvedOxygenFieldName);
              updateDataVariableUnits(self, '\muM');
            
            % Derive dissolved oxygen and oxygen saturation values from voltage (membrane sensors)  
          else if ( any(strcmp(self.configuration.dissolvedOxygenVoltsFieldName,   ...
                                      self.configuration.dataVariableNames)) && ...
                      length(self.configuration.dataVariableNames) ==                ...
                      length(self.configuration.dataVariableUnits) )
                  
              % coefficient constants             
              
              % get a reference to the coefficient constants for the DO
              % sensor
              dissolvedOxygen_Soc = ...
                self.configuration.calibrationCoefficientValues{ ...
                  find( ...
                    strcmp( ...
                      'dissolvedOxygen_Soc', ...
                      self.configuration.calibrationCoefficientNames   ...
                    ) ...
                  ) ...
                };  
            
            dissolvedOxygen_Voffset = ...
                self.configuration.calibrationCoefficientValues{ ...
                  find( ...
                    strcmp( ...
                      'dissolvedOxygen_Voffset', ...
                      self.configuration.calibrationCoefficientNames   ...
                    ) ...
                  ) ...
                };   
            
            dissolvedOxygen_Tau20 = ...
                self.configuration.calibrationCoefficientValues{ ...
                  find( ...
                    strcmp( ...
                      'dissolvedOxygen_Tau20', ...
                      self.configuration.calibrationCoefficientNames   ...
                    ) ...
                  ) ...
                };   
              
            dissolvedOxygen_A = ...
                self.configuration.calibrationCoefficientValues{ ...
                  find( ...
                    strcmp( ...
                      'dissolvedOxygen_A', ...
                      self.configuration.calibrationCoefficientNames   ...
                    ) ...
                  ) ...
                };   
            
            dissolvedOxygen_B = ...
                self.configuration.calibrationCoefficientValues{ ...
                  find( ...
                    strcmp( ...
                      'dissolvedOxygen_B', ...
                      self.configuration.calibrationCoefficientNames   ...
                    ) ...
                  ) ...
                };   
            
            dissolvedOxygen_C = ...
                self.configuration.calibrationCoefficientValues{ ...
                  find( ...
                    strcmp( ...
                      'dissolvedOxygen_C', ...
                      self.configuration.calibrationCoefficientNames   ...
                    ) ...
                  ) ...
                };   
            
            dissolvedOxygen_Enom = ...
                self.configuration.calibrationCoefficientValues{ ...
                  find( ...
                    strcmp( ...
                      'dissolvedOxygen_Enom', ...
                      self.configuration.calibrationCoefficientNames   ...
                    ) ...
                  ) ...
                };
            
            % get a reference to the dissolved oxygen voltage array
              dissolvedOxygen_V0 = ...
                self.dataCellArray{ ...
                  find( ...
                    strcmp( ...
                      self.configuration.dissolvedOxygenVoltsFieldName, ...
                      self.configuration.dataVariableNames ...
                    ) ...
                  ) ...
                };
            
            T0=24.4109;
            Ta=T0+273.15;
            S0=35.00;
            P0=1;
            OxSol=4.774;
            
            DO=dissolvedOxygen_Soc*(dissolvedOxygen_V0 + dissolvedOxygen_Voffset)* ...
                   (1.0 + dissolvedOxygen_A*T0 + dissolvedOxygen_B*T0^2 + dissolvedOxygen_C*T0^3)* ...
                   OxSol*exp(dissolvedOxygen_Enom*P0/Ta);
            
            value=DO * 44.66;
            
            clear  dissolvedOxygen_A dissolvedOxygen_B dissolvedOxygen_C     ...
                    dissolvedOxygen_Soc dissolvedOxygen_Voffset dissolvedOxygen_Tau20  ...
                    dissolvedOxygen_Enom dissolvedOxygen_V0 T0 Ta S0 P0 OxSol   ...
                    DO
            
            
            % update the field names and units arrays
              updateDataVariableNames(self, self.configuration.dissolvedOxygenFieldName);
              updateDataVariableUnits(self, '\muM');
            
            
            % Derive dissolved oxygen from DO phase and thermistor values
            % (optical sensors)
            else if ( any(strcmp(self.configuration.dissolvedOxygenPhaseFieldName,   ...
                                      self.configuration.dataVariableNames)) && ...
                 any(strcmp(self.configuration.thermistorVoltageFieldName,      ...
                                      self.configuration.dataVariableNames)) && ...                     
                 length(self.configuration.dataVariableNames) ==                ...
                 length(self.configuration.dataVariableUnits) )    
              
              % coefficient constants             
              
              % get a reference to the coefficient constants for the DO
              % sensor
              dissolvedOxygen_A0 = ...
                self.configuration.calibrationCoefficientValues{ ...
                  find( ...
                    strcmp( ...
                      'dissolvedOxygen_A0', ...
                      self.configuration.calibrationCoefficientNames   ...
                    ) ...
                  ) ...
                };

              dissolvedOxygen_A1 = ...
                self.configuration.calibrationCoefficientValues{ ...
                  find( ...
                    strcmp( ...
                      'dissolvedOxygen_A1', ...
                      self.configuration.calibrationCoefficientNames   ...
                    ) ...
                  ) ...
                };
            
            dissolvedOxygen_A2 = ...
                self.configuration.calibrationCoefficientValues{ ...
                  find( ...
                    strcmp( ...
                      'dissolvedOxygen_A2', ...
                      self.configuration.calibrationCoefficientNames   ...
                    ) ...
                  ) ...
                };
            
            dissolvedOxygen_Ta0 = ...
                self.configuration.calibrationCoefficientValues{ ...
                  find( ...
                    strcmp( ...
                      'dissolvedOxygen_Ta0', ...
                      self.configuration.calibrationCoefficientNames   ...
                    ) ...
                  ) ...
                };
            
            dissolvedOxygen_Ta1 = ...
                self.configuration.calibrationCoefficientValues{ ...
                  find( ...
                    strcmp( ...
                      'dissolvedOxygen_Ta1', ...
                      self.configuration.calibrationCoefficientNames   ...
                    ) ...
                  ) ...
                };
            
            dissolvedOxygen_Ta2 = ...
                self.configuration.calibrationCoefficientValues{ ...
                  find( ...
                    strcmp( ...
                      'dissolvedOxygen_Ta2', ...
                      self.configuration.calibrationCoefficientNames   ...
                    ) ...
                  ) ...
                };
            
            dissolvedOxygen_Ta3 = ...
                self.configuration.calibrationCoefficientValues{ ...
                  find( ...
                    strcmp( ...
                      'dissolvedOxygen_Ta3', ...
                      self.configuration.calibrationCoefficientNames   ...
                    ) ...
                  ) ...
                };
            
            dissolvedOxygen_B0 = ...
                self.configuration.calibrationCoefficientValues{ ...
                  find( ...
                    strcmp( ...
                      'dissolvedOxygen_B0', ...
                      self.configuration.calibrationCoefficientNames   ...
                    ) ...
                  ) ...
                };
            
            dissolvedOxygen_B1 = ...
                self.configuration.calibrationCoefficientValues{ ...
                  find( ...
                    strcmp( ...
                      'dissolvedOxygen_B1', ...
                      self.configuration.calibrationCoefficientNames   ...
                    ) ...
                  ) ...
                };
            
            dissolvedOxygen_C0 = ...
                self.configuration.calibrationCoefficientValues{ ...
                  find( ...
                    strcmp( ...
                      'dissolvedOxygen_C0', ...
                      self.configuration.calibrationCoefficientNames   ...
                    ) ...
                  ) ...
                };
            
            dissolvedOxygen_C1 = ...
                self.configuration.calibrationCoefficientValues{ ...
                  find( ...
                    strcmp( ...
                      'dissolvedOxygen_C1', ...
                      self.configuration.calibrationCoefficientNames   ...
                    ) ...
                  ) ...
                };
            
            dissolvedOxygen_C2 = ...
                self.configuration.calibrationCoefficientValues{ ...
                  find( ...
                    strcmp( ...
                      'dissolvedOxygen_C2', ...
                      self.configuration.calibrationCoefficientNames   ...
                    ) ...
                  ) ...
                };
            
            dissolvedOxygen_E = ...
                self.configuration.calibrationCoefficientValues{ ...
                  find( ...
                    strcmp( ...
                      'dissolvedOxygen_E', ...
                      self.configuration.calibrationCoefficientNames   ...
                    ) ...
                  ) ...
                };
            
              % get a reference to the coefficient constants for salinity
              salinity_B0 = ...
                self.configuration.calibrationCoefficientValues{ ...
                  find( ...
                    strcmp( ...
                      'salinity_B0', ...
                      self.configuration.calibrationCoefficientNames   ...
                    ) ...
                  ) ...
                };
            
            salinity_B1 = ...
                self.configuration.calibrationCoefficientValues{ ...
                  find( ...
                    strcmp( ...
                      'salinity_B1', ...
                      self.configuration.calibrationCoefficientNames   ...
                    ) ...
                  ) ...
                };
            
            salinity_B2 = ...
                self.configuration.calibrationCoefficientValues{ ...
                  find( ...
                    strcmp( ...
                      'salinity_B2', ...
                      self.configuration.calibrationCoefficientNames   ...
                    ) ...
                  ) ...
                };
            
            salinity_B3 = ...
                self.configuration.calibrationCoefficientValues{ ...
                  find( ...
                    strcmp( ...
                      'salinity_B3', ...
                      self.configuration.calibrationCoefficientNames   ...
                    ) ...
                  ) ...
                };
            
            salinity_C0 = ...
                self.configuration.calibrationCoefficientValues{ ...
                  find( ...
                    strcmp( ...
                      'salinity_C0', ...
                      self.configuration.calibrationCoefficientNames   ...
                    ) ...
                  ) ...
                };

              % get a reference to the dissolved oxygen phase array
              dissolvedOxygenPhaseArray = ...
                self.dataCellArray{ ...
                  find( ...
                    strcmp( ...
                      self.configuration.dissolvedOxygenPhaseFieldName, ...
                      self.configuration.dataVariableNames ...
                    ) ...
                  ) ...
                };
            
             % get a reference to the thermistor voltage array
              thermistorVoltsArray = ...
                self.dataCellArray{ ...
                  find( ...
                    strcmp( ...
                      self.configuration.thermistorVoltageFieldName, ...
                      self.configuration.dataVariableNames ...
                    ) ...
                  ) ...
                };
              
             % get a reference to the salinity array
              salinityArray = ...
                self.dataCellArray{ ...
                  find( ...
                    strcmp( ...
                      self.configuration.salinityFieldName, ...
                      self.configuration.dataVariableNames ...
                    ) ...
                  ) ...
                };
            
            % Calculate variables used to derive DO concentration
             Lv = log(100000*thermistorVoltsArray ./ (3.3-thermistorVoltsArray)) ; % log voltage
            
             VDO = dissolvedOxygenPhaseArray / 39.457071                         ; % DO voltage
             
             Ta = 1 ./ ( dissolvedOxygen_Ta0                      ...
                         + dissolvedOxygen_Ta1 .* Lv              ...
                         + dissolvedOxygen_Ta2 .* Lv.^2           ...
                         + dissolvedOxygen_Ta3 .* Lv.^3 )                        ; % absolute temperature
             
             T0 = Ta - 273.15                                                    ; % Temperature (Celcius)
             
             P0 = 1                                                              ; % pressure (decibars)
             
             Ksv = dissolvedOxygen_C0                             ...
                   + dissolvedOxygen_C1 * T0                      ...
                   + dissolvedOxygen_C2 * T0.^2                                  ; % Stern-Volmer Constant Eqn
             
             Ts = log( (298.15 - T0) ./ Ta )                                     ;
             
             Scorr = exp( salinityArray                           ...
                          .* ( salinity_B0                        ...
                          + salinity_B1 * Ts                      ...
                          + salinity_B2 * Ts.^2                   ...
                          + salinity_B3 * Ts.^3 )                 ...
                        + salinity_C0 * salinityArray.^2 )                       ; % Salinity Correlation
                    
             Pcorr = exp( dissolvedOxygen_E * P0 ./ Ta)                          ; % Pressure Correlation
           
            % calculate dissolved oxygen concentration in mL/L
             DO = (( ( dissolvedOxygen_A0                         ...
                        + dissolvedOxygen_A1 * T0                 ...
                        + dissolvedOxygen_A2 * VDO.^2 )           ...
                      ./ ( dissolvedOxygen_B0                     ...
                           + dissolvedOxygen_B1 * VDO )           ...
                      - 1.0 )                                     ...
                    ./ Ksv                                        ...
                    .* Scorr .* Pcorr )                                          ; % Dissolved oxygen (mL/L)
            % calculate dissolved oxygen concentration in micro Moles
              value = DO * 44.66                                                 ; % Dissolved oxygen (micro Molar)   
           
                           
              clear dissolvedOxygen_A0 dissolvedOxygen_A1 dissolvedOxygen_A2     ...
                    dissolvedOxygen_Ta0 dissolvedOxygen_Ta1 dissolvedOxygen_Ta2  ...
                    dissolvedOxygen_Ta3 dissolvedOxygen_B0 dissolvedOxygen_B1    ...
                    dissolvedOxygen_C0 dissolvedOxygen_C1 dissolvedOxygen_C2     ...
                    dissolvedOxygen_E salinity_B0 salinity_B1 salinity_B2        ...
                    salinity_B3 salinity_C0 Lv VDO T0 P0 Ksv Ts Scorr Pcorr      ...
                    dissolvedOxygenPhaseArray thermistorVoltsArray DO Ta salinityArray
              
              % update the field names and units arrays
              updateDataVariableNames(self, self.configuration.dissolvedOxygenFieldName);
              updateDataVariableUnits(self, '\muM');

              else
                error(['There is no dissolvedOxygenPhase field designated ' ...
                       'in the dataVariableNames cell array.  Failed to build ' ...
                       'dissolvedOxygen vector. ' ...
                       'Set the dataVariableNames property and the dataVariableUnits ' ...
                       'property to include "dissolvedOxygenPhase" name ' ...
                       'and their respective format strings.']);              
                end 
              end
            end
          else
              value = [];
          end % end if statement (~isempty(self.dataCellArray))
          
% derive oxygen saturation from thermistor values
        case self.configuration.oxygenSaturationFieldName
          if ( ~isempty(self.dataCellArray{ ...
                  find( ...
                    strcmp( ...
                      self.configuration.dissolvedOxygenFieldName, ...
                      self.configuration.dataVariableNames ...
                    ) ...
                  ) ...
                } ...
               ) ) 
            if ( any(strcmp(self.configuration.dissolvedOxygenPhaseFieldName,   ...
                                      self.configuration.dataVariableNames)) && ...
                 any(strcmp(self.configuration.thermistorVoltageFieldName,      ...
                                      self.configuration.dataVariableNames)) && ...                     
                 length(self.configuration.dataVariableNames) ==                ...
                 length(self.configuration.dataVariableUnits) )
             
            % get a reference to the coefficient constants to derive
            % oxygen solubility
            solubility_A1 = ...
               self.configuration.calibrationCoefficientValues{ ...
                 find( ...
                   strcmp( ...
                     'solubility_A1', ...
                     self.configuration.calibrationCoefficientNames   ...
                   ) ...
                 ) ...
               };
           
           solubility_A2 = ...
               self.configuration.calibrationCoefficientValues{ ...
                 find( ...
                   strcmp( ...
                     'solubility_A2', ...
                     self.configuration.calibrationCoefficientNames   ...
                   ) ...
                 ) ...
               };
           
           solubility_A3 = ...
               self.configuration.calibrationCoefficientValues{ ...
                 find( ...
                   strcmp( ...
                     'solubility_A3', ...
                     self.configuration.calibrationCoefficientNames   ...
                   ) ...
                 ) ...
               };
           
           solubility_A4 = ...
               self.configuration.calibrationCoefficientValues{ ...
                 find( ...
                   strcmp( ...
                     'solubility_A4', ...
                     self.configuration.calibrationCoefficientNames   ...
                   ) ...
                 ) ...
               };
           
           solubility_B1 = ...
               self.configuration.calibrationCoefficientValues{ ...
                 find( ...
                   strcmp( ...
                     'solubility_B1', ...
                     self.configuration.calibrationCoefficientNames   ...
                   ) ...
                 ) ...
               };
           
           solubility_B2 = ...
               self.configuration.calibrationCoefficientValues{ ...
                 find( ...
                   strcmp( ...
                     'solubility_B2', ...
                     self.configuration.calibrationCoefficientNames   ...
                   ) ...
                 ) ...
               };
           
           solubility_B3 = ...
               self.configuration.calibrationCoefficientValues{ ...
                 find( ...
                   strcmp( ...
                     'solubility_B3', ...
                     self.configuration.calibrationCoefficientNames   ...
                   ) ...
                 ) ...
               };
           
           dissolvedOxygen_Ta0 = ...
                self.configuration.calibrationCoefficientValues{ ...
                  find( ...
                    strcmp( ...
                      'dissolvedOxygen_Ta0', ...
                      self.configuration.calibrationCoefficientNames   ...
                    ) ...
                  ) ...
                };
            
            dissolvedOxygen_Ta1 = ...
                self.configuration.calibrationCoefficientValues{ ...
                  find( ...
                    strcmp( ...
                      'dissolvedOxygen_Ta1', ...
                      self.configuration.calibrationCoefficientNames   ...
                    ) ...
                  ) ...
                };
            
            dissolvedOxygen_Ta2 = ...
                self.configuration.calibrationCoefficientValues{ ...
                  find( ...
                    strcmp( ...
                      'dissolvedOxygen_Ta2', ...
                      self.configuration.calibrationCoefficientNames   ...
                    ) ...
                  ) ...
                };
            
            dissolvedOxygen_Ta3 = ...
                self.configuration.calibrationCoefficientValues{ ...
                  find( ...
                    strcmp( ...
                      'dissolvedOxygen_Ta3', ...
                      self.configuration.calibrationCoefficientNames   ...
                    ) ...
                  ) ...
                };
           
           % get a reference to the thermistor voltage array
              thermistorVoltsArray = ...
                self.dataCellArray{ ...
                  find( ...
                    strcmp( ...
                      self.configuration.thermistorVoltageFieldName, ...
                      self.configuration.dataVariableNames ...
                    ) ...
                  ) ...
                };
            
            % get a reference to the salinity array
              salinityArray = ...
                self.dataCellArray{ ...
                  find( ...
                    strcmp( ...
                      self.configuration.salinityFieldName, ...
                      self.configuration.dataVariableNames ...
                    ) ...
                  ) ...
                };
            
            % get a reference to the dissolved oxygen array
              dissolvedOxygenArray = ...
                self.dataCellArray{ ...
                  find( ...
                    strcmp( ...
                      self.configuration.dissolvedOxygenFieldName, ...
                      self.configuration.dataVariableNames ...
                    ) ...
                  ) ...
                };
           
           % Calculate variables used to derive DO concentration
             Lv = log(100000*thermistorVoltsArray ./ (3.3-thermistorVoltsArray)) ; % log voltage
             
             Ta = 1 ./ ( dissolvedOxygen_Ta0                      ...
                         + dissolvedOxygen_Ta1 .* Lv              ...
                         + dissolvedOxygen_Ta2 .* Lv.^2           ...
                         + dissolvedOxygen_Ta3 .* Lv.^3 )                        ; % absolute temperature
           
          % Calculate oxygen solubility
           oxygenSolubility = exp(                                    ...
                                  ( solubility_A1                     ...
                                   + solubility_A2 * 100 ./ Ta        ...
                                   + solubility_A3 * log( Ta / 100 )  ...
                                   + solubility_A4 * Ta / 100 )       ...
                                  + salinityArray                     ...
                                   .* ( solubility_B1                 ...
                                     + solubility_B2 * Ta / 100       ...
                                     + solubility_B3 * ( Ta / 100 ).^2 ) )  ;
                                 
           value = ( dissolvedOxygenArray / 44.66 )  ...
                     ./ oxygenSolubility * 100                 ;
             
           clear solubility_A1 solubility_A2 solubility_A3 solubility_A4     ...
                 solubility_B1 solubility_B2 solubility_B3 Ta salinityArray  ...
                 oxygenSolubility
             
             % update the field names and units arrays
              updateDataVariableNames(self, self.configuration.oxygenSaturationFieldName);
              updateDataVariableUnits(self, '%');
             
            else if ( any(strcmp(self.configuration.dissolvedOxygenVoltsFieldName,   ...
                                      self.configuration.dataVariableNames)) && ...                    
                 length(self.configuration.dataVariableNames) ==                ...
                 length(self.configuration.dataVariableUnits) )
             
             % get a reference to the dissolved oxygen array
              dissolvedOxygenArray = ...
                self.dataCellArray{ ...
                  find( ...
                    strcmp( ...
                      self.configuration.dissolvedOxygenFieldName, ...
                      self.configuration.dataVariableNames ...
                    ) ...
                  ) ...
                };
            
            OxSol=4.774;
            value=100*dissolvedOxygenArray/44.66/OxSol;
             
            clear OxSol
            
           % update the field names and units arrays
              updateDataVariableNames(self, self.configuration.oxygenSaturationFieldName);
              updateDataVariableUnits(self, '%');

            else
              error(['There is no dissolvedOxygenPhase field designated ' ...
                     'in the dataVariableNames cell array.  Failed to build ' ...
                     'dissolvedOxygen vector. ' ...
                     'Set the dataVariableNames property and the dataVariableUnits ' ...
                     'property to include "dissolvedOxygenPhase" name ' ...
                     'and their respective format strings.']); 
                end
            end
          else
              value = [];
          end % end if statement (~isempty(self.dataCellArray))          
      
      end % end switch
      
    end % end createDerivedVariable function
    
    % A method to create an adjusted depth variable corrected to MLLW based
    % on input from the schedule script. Appends the resulting column to
    % the data cell array.
    % @returns value - variable array
    function value = depthAdjustment(self,time)
        
    if ( self.configuration.debug )
        disp('DataProcessor.depthAdjustment() called.');
    end

    index=find(strcmp('depth',self.configuration.dataVariableNames));
    if index > 0
        depth=self.dataCellArray{index};
        if ~isempty(self.configuration.MLLWadjustmentDate) && ...
                length(self.configuration.MLLWadjustment)>1
           adjustedDepth=depth;
           t1=find(time<datenum(char(self.configuration.MLLWadjustmentDate)));
           adjustedDepth(t1)=depth(t1) + self.configuration.MLLWadjustment(1);
           t2=find(time>=datenum(char(self.configuration.MLLWadjustmentDate)));
           adjustedDepth(t2)=depth(t2) + self.configuration.MLLWadjustment(2);
        else
           adjustedDepth=depth + self.configuration.MLLWadjustment;         %correct to MLLW
        end
       value = adjustedDepth;
    
     % update the field names and units arrays
       updateDataVariableNames(self, 'adjustedDepth');
       updateDataVariableUnits(self, 'm');

    else
       value = [];    
    end
end

    % A method used to create a Temperature/Salinity figure.
    % @returns value - the handle to the figure object that is created  
    function value = createTSFigure(self                   , ...
                                    figureTitlePrefix      , ...
                                    figureStartDate        , ...
                                    figureDuration         , ...
                                    figureYAxisVariables   , ...
                                    figureXAxisVariables   , ...
                                    figureXAxisTickStep    , ...
                                    plotTickFormats        , ...
                                    graphicMarkers         , ...
                                    graphicMarkersizes     , ...
                                    graphicMarkerColors    , ...
                                    includeMovingAverage   , ...
                                    movingAverageDuration  , ...
                                    movingAverageLineColor , ...
                                    movingAverageLineWidth   ...
    )
      if ( self.configuration.debug )
        disp('DataProcessor.createTSFigure() called.');
      end
      
      temperatureArray = self.dataCellArray{1};  % temperature
      salinityArray = self.dataCellArray{5};     % salinity
      serialDateArray = self.dataCellArray{7};   % serialdate
      
      close(gcf);
      figureHandle     = figure(); clf;
      numberOfPlots    = length(figureYAxisVariables{1});
      figureRectangle  = [0 800 1296 800];
      paperPosition    = [0 0 8.5 11.0];
      yLabelPosition   = [-0.1 0.5042 0];
      figureHandle     = figure(); clf;
      figureDurationInDays = figureDuration/60/60/24;

      % iterate through the number of figures and plot them
      for plotNumber = 1:numberOfPlots
        
        % get the variable label based on the Y axis variable list passed to 
        % this function
        variableLabel = char(figureYAxisVariables{1}(plotNumber));
        % lookup up the units based on the variable name
        unitLabel = ...
          self.configuration.dataVariableUnits{ ...
            find( ...
              strcmp( ...
                variableLabel, self.configuration.dataVariableNames ...
              ) ...
            ) ...
          };
        
        % build the subplot's Y label string
        figureYAxisLabel = [variableLabel ' ' '(' unitLabel ')'];
        
        variableLabel = char(figureXAxisVariables{1}(plotNumber));
        % lookup up the units based on the variable name
        unitLabel = ...
          self.configuration.dataVariableUnits{ ...
            find( ...
              strcmp( ...
                variableLabel, self.configuration.dataVariableNames ...
              ) ...
            ) ...
          };
        
        % build the subplot's X label string
        figureXAxisLabel = [variableLabel ' ' '(' unitLabel ')'];
        
        % get yearly, weekly, daily, and current time indices
        currentYearIndices = find( serialDateArray < max(serialDateArray) - 7 );
        currentWeekIndices = intersect(find( serialDateArray > max(serialDateArray) - 7), ...
                                   find(serialDateArray < max(serialDateArray) - 1));
        currentDayIndices = find( serialDateArray > max(serialDateArray) - 1 );
        currentObservationIndex = find(serialDateArray == max(serialDateArray));
        disp(['currentObservationIndex is ' num2str(currentObservationIndex)]);
        disp(['currentObservation Date is ' datestr(max(serialDateArray))]);
        
        % plot the current year data
        if ( ~isempty(currentYearIndices) )
          currentYearPlot = ...
          plot(salinityArray(currentYearIndices),            ...
               temperatureArray(currentYearIndices),         ...
               'LineStyle', 'none',                          ...
               'Marker', '.',                                ...
               'MarkerSize', 2,                              ...
               'MarkerFaceColor', [200/255 200/255 200/255], ...
               'MarkerEdgeColor', [200/255 200/255 200/255]  ...
              );
          currentYearTimeRange =                               ...
            [datestr(serialDateArray(min(currentYearIndices)), ...
                     'mm/dd/yy')                               ...
             ' - '                                             ...
             datestr(serialDateArray(max(currentYearIndices)), ...
                     'mm/dd/yy')];
        end
        
        % overlay the current week data
        set(gca, 'NextPlot', 'add');
        if ( ~isempty(currentWeekIndices) )
          currentWeekPlot = ...
          plot(salinityArray(currentWeekIndices),            ...
               temperatureArray(currentWeekIndices),         ...
               'LineStyle', 'none',                          ...
               'Marker', '.',                                ...
               'MarkerSize', 3,                              ...
               'MarkerFaceColor', [000/255 000/255 255/255], ...
               'MarkerEdgeColor', [000/255 000/255 255/255]  ...
              );
        currentWeekTimeRange =                               ...
          [datestr(serialDateArray(min(currentWeekIndices)), ...
                   'mm/dd/yy')                               ...
           ' - '                                             ...
           datestr(serialDateArray(max(currentWeekIndices)), ...
                   'mm/dd/yy')];
          
        end
        
        % overlay the current day data
        set(gca, 'NextPlot', 'add');
        if ( ~isempty(currentDayIndices) )
          currentDayPlot = ...
          plot(salinityArray(currentDayIndices),              ...
               temperatureArray(currentDayIndices),           ...
               'LineStyle', 'none',                           ...
               'Marker', '.',                                 ...
               'MarkerSize', 3,                               ...
               'MarkerFaceColor', [000/255 255/255 000/255],  ...
               'MarkerEdgeColor', [000/255 255/255 000/255]   ...
              );
          currentDayTimeRange =                               ...
            [datestr(serialDateArray(min(currentDayIndices)), ...
                     'mm/dd/yy HH:MM')                        ...
             ' - '                                            ...
             datestr(serialDateArray(max(currentDayIndices)), ...
                     'mm/dd/yy HH:MM')];
          
        end
        
        % overlay the current observation data
        set(gca, 'NextPlot', 'add');
        if ( ~isempty(currentObservationIndex) )
          currentObservationPlot = ...
          plot(salinityArray(currentObservationIndex), ...
               temperatureArray(currentObservationIndex), ...
               'LineStyle', 'none',             ...
               'LineWidth', 1.0,                ...
               'Marker', 'o',                   ...
               'MarkerSize', 6,                 ...
               'MarkerFaceColor', 'none',       ...
               'MarkerEdgeColor', [255/255 0 0] ...
              );
          currentObservationTime =                                 ...
            datestr(serialDateArray(min(currentObservationIndex)), ...
                    'mm/dd/yy HH:MM');
          
        end
        
        % Add the plot title
        titleHandle  =                                             ...
          title ([figureTitlePrefix                                ...
                 ' for: '                                          ...
                 datestr(min(serialDateArray), 'mm/dd/yyyy HH:MM') ...
                 ' - '                                             ...
                 datestr(max(serialDateArray), 'mm/dd/yyyy HH:MM') ...
                 ' (HST)']);
        
        % Add the plot axes labels
        xLabelHandle = xlabel(figureXAxisLabel);
        yLabelHandle = ylabel(figureYAxisLabel);
        
        % Add axes styling
        set(gca, ...
          'Box'         , 'off'                                 , ...
          'TickDir'     , 'out'                                , ...
          'TickLength'  , [.01 .01]                            , ...
          'XMinorTick'  , 'on'                                 , ...
          'YMinorTick'  , 'on'                                 , ...
          'YTickLabel'  , num2str(get(gca, 'Ytick')', '%3.2f') , ...
          'XTickLabel'  , num2str(get(gca, 'Xtick')', '%3.2f') , ...
          'XGrid'       , 'on'                                 , ...
          'YGrid'       , 'on'                                 , ...
          'XColor'      , [.3 .3 .3]                           , ...
          'YColor'      , [.3 .3 .3]                           , ...
          'LineWidth'   , 1 );

        % Add the legend and increase legend point marker size
        legendInfo = '';
        if ( exist('currentYearPlot') )
          legendInfo = [legendInfo '''yearly observations  (' currentYearTimeRange ')'','];
        end

        if ( exist('currentWeekPlot') )
          legendInfo = [legendInfo '''weekly observations (' currentWeekTimeRange ')'','];
        end

        if ( exist('currentDayPlot') )
          legendInfo = [legendInfo '''daily observations     (' currentDayTimeRange ')'','];
        end

        if ( exist('currentObservationPlot') )
          legendInfo = [legendInfo '''latest observation      (' currentObservationTime ')'','];
        end
        legendInfo = [legendInfo '''Location'', ''NorthWest'''];
        legendHandle = ...
        eval(['legend(' legendInfo ')']);

        % Increase point marker size in the legend
        legendChildren = get(legendHandle, 'Children');
        for child = 1:length(legendChildren)
          try
            if ( ~isempty(strmatch(get(legendChildren(child), 'Marker'), '.')) )
              set(legendChildren(child), 'MarkerSize', 20);
            end
          catch
            continue;
          end
        end % for child
        
      end % for plotNumber
      value = figureHandle;
      
    end % createTSFigure()
    
    % A method used to create a times series figure with subplots.
    % @returns value - the handle to the figure object that is created  
    function value = createTimesSeriesFigure(          ...
                              self                   , ...
                              figureTitlePrefix      , ...
                              figureStartDate        , ...
                              figureDuration         , ...
                              figureYAxisVariables   , ...
                              figureXAxisVariables   , ...
                              figureXAxisTickStep    , ...
                              plotTickFormats        , ...
                              graphicMarkers         , ...
                              graphicMarkersizes     , ...
                              graphicMarkerColors    , ...
                              includeMovingAverage   , ...
                              movingAverageDuration  , ...
                              movingAverageLineColor , ...
                              movingAverageLineWidth   ...
    )
      if ( self.configuration.debug )
        disp('DataProcessor.createTimesSeriesFigure() called.');
      end

      %============
      % Plotting
      %============
      close(gcf);
      numberOfPlots = length(figureYAxisVariables{1});
      figureRectangle = [0 0 1200 800];
      paperPosition = [0 0 11.0 8.5];
      yLabelPosition = [-0.1 0.5042 0];
      figureHandle = figure(); clf;
      figureDurationInDays = figureDuration/60/60/24;

      % iterate through the number of figures and plot them
      for plotNumber = 1:numberOfPlots
        
        % get the variable label based on the Y axis variable list passed to 
        % this function
        variableLabel = char(figureYAxisVariables{1}(plotNumber));
        % lookup up the units based on the variable name
        unitLabel = ...
          self.configuration.dataVariableUnits{ ...
            find( ...
              strcmp( ...
                variableLabel, self.configuration.dataVariableNames ...
              ) ...
            ) ...
          };
        
        % build the subplot's Y label string
        figureYAxisLabel = [variableLabel ' ' '(' unitLabel ')'];
        
        x=[]; y=[];
        
        % Position and Size of Figure
        set(gcf, 'units', 'pixels',              ...
                 'position', figureRectangle,    ...
                 'paperposition', paperPosition);
        
        % Plotting
        % build the X and Y variables
        xAxisVariableName = char(figureXAxisVariables{1}(1));
        x = self.dataCellArray{ ...
              find( ...
                strcmp( ...
                  xAxisVariableName, ...
                  self.configuration.dataVariableNames ...
                ) ...
              ) ...
            };
        
        yAxisVariableName = char(figureYAxisVariables{1}(plotNumber));
        y = self.dataCellArray{ ...
              find( ...
                strcmp( ...
                  yAxisVariableName, ...
                  self.configuration.dataVariableNames ...
                ) ...
              ) ...
            };
        
        % Subset the y and x axis data based on the duration of the
        % particular figure. Order matters because y depends on x
        if ( strcmp(self.configuration.reference, 'newest') )
          % for 'most recent' data duration
          y = y(find(x >= (max(x) - figureDurationInDays)));          
          x = x(find(x >= (max(x) - figureDurationInDays)));          
          
          % flip the vectors in order to append to them
          xFlipped = flipud(x);
          yFlipped = flipud(y);
          
          if ( min(xFlipped) > max(xFlipped) - figureDurationInDays )
            xFlipped(find(xFlipped == min(xFlipped)) + 1) = max(x) - figureDurationInDays;
            yFlipped(length(yFlipped) + 1) = NaN;
          end
          
          x = flipud(xFlipped);
          y = flipud(yFlipped);
          
        else
          HSTOffsetInDays = 10/24; % Hawaii time hour offset in days
          startSerialDate = datenum(figureStartDate) - HSTOffsetInDays;
          % for 'fixed' start date  ('absolute' or 'oldest' data reference)
          y = y(find(x >= startSerialDate));
          x = x(find(x >= startSerialDate));
          y = y(find(x <= (startSerialDate + figureDurationInDays)));
          x = x(find(x <= startSerialDate + figureDurationInDays));
          
        end
        
        % get the graphic color
        graphicColorArray = graphicMarkerColors{1};
        graphicMarkerColor = graphicColorArray{plotNumber};
        
        % get the graphic marker
        graphicMarkerArray = graphicMarkers{1};
        graphicMarker      = graphicMarkerArray{plotNumber};
        
        % get the graphic marker
        graphicMarkersizeArray = graphicMarkersizes{1};
        graphicMarkersize      = graphicMarkersizeArray{plotNumber};
                
        subplot(numberOfPlots, 1, plotNumber);
        plotHandle = plot(x, y,                                              ...
                          'color'     , graphicMarkerColor,                  ...
                          'LineStyle' , 'none',                              ...
                          'Marker'    , graphicMarker,                       ...
                          'MarkerSize', graphicMarkersize);
        
        % Add in the moving average line if it should be included
        if ( includeMovingAverage )
          set(gca, 'NextPlot', 'add');
          plot(x, moving(y, movingAverageDuration)  , ...
               'color'    , movingAverageLineColor{1}  , ...
               'linewidth', movingAverageLineWidth    ...
          );
          
          % Add the legend and increase legend point marker size
          legendHandle = ...
          legend([yAxisVariableName ' observations']                   , ...
                 [num2str(movingAverageDuration/60) ' minute average'] , ...
                 'Location', 'NorthWest');

          % Increase point marker size in the legend
          legendChildren = get(legendHandle, 'Children');
          for child = 1:length(legendChildren)
            try
              if ( ~isempty(strmatch(get(legendChildren(child), 'Marker'), '.')) )
                set(legendChildren(child), 'MarkerSize', 20);
              end
            catch
              continue;
            end
          end
          
        end
        
        % Add the figure title to the top of the figure only
        
        latestObservationTime   = max(x);
        earliestObservationTime = round(min(x)) + 1;
        
        % round the observation period to the nearest timerInterval (i.e even 20 minute periods)
        latestSeconds = str2num(datestr(latestObservationTime, 'SS'));
        latestMinutes = str2num(datestr(latestObservationTime, 'MM'));
        latestHour    = str2num(datestr(latestObservationTime, 'HH'));
        latestDay     = str2num(datestr(latestObservationTime, 'dd'));
        latestMonth   = str2num(datestr(latestObservationTime, 'mm'));
        latestYear    = str2num(datestr(latestObservationTime, 'yyyy'));
        if ( latestMinutes < 5)
          latestMinutes = latestMinutes + 10;
        end
        
        latestMinutes = round(latestMinutes/10) * 10;
        remainder = mod(latestMinutes, self.configuration.timerInterval);
        latestMinutes = latestMinutes + remainder;
        
        if ( latestMinutes == 60 )
          latestMinutes = 0;
          latestHour = latestHour + 1;
        end
        
        % reset the latest observation time to the time rounded to the timerInterval
        %latestObservationTime = datenum([latestYear latestMonth latestDay latestHour latestMinutes 0]);
        
        if ( plotNumber == 1 )
          titleText = [figureTitlePrefix ' : '         ...
                      datestr(earliestObservationTime, ...
                      'mm-dd-yyyy HH:MM') ' to '       ...
                      datestr(latestObservationTime,'mm-dd-yyyy HH:MM')];
          title(titleText);
        end
        grid on; axhan = gca; 
        
        % Limits
        % set the X axis limits based on the figure duration passed into
        % this function
        
        % set the X limits for single day plots
        xlim([round(min(x)) + 1 round(max(x)) + 1]);        
        
        % set the Y axis limits based on min and max observations
        minYObservation=[]; maxYObservation=[];
        minYObservation=min(y); maxYObservation=max(y);
        % Set the ylimits only if the range is increasing 
        if ( minYObservation < maxYObservation )
          ylim([minYObservation maxYObservation]);
        end
      
        
        % Ticks and Ticklabels
        %xtick = get(axhan,'XTick');
        xtick = ((round(max(x)) + 1) - (figureDurationInDays):figureXAxisTickStep:(round(max(x) + 1)));
        if ( figureDurationInDays <= 3 )
          xticklabel = datestr(xtick,'mm/dd/yy HH:MM');
        else  
          xticklabel = datestr(xtick,'mm/dd/yy');
        end
        set(axhan, ...
            'XColor'      , [.3 .3 .3], ...
            'YColor'      , [.3 .3 .3]  ...
        );
        % get the figure tick format
        plotTickFormatArray = plotTickFormats{1};
        plotTickFormat      = plotTickFormatArray{plotNumber};
        
        ytick = get(axhan,'YTick');
        yticklabel = num2str(ytick',plotTickFormat);
        set(axhan,'XTick',xtick,'YTick',ytick);
        set(axhan,'XTickLabel',xticklabel,'YTickLabel',yticklabel);
        
        %Add Fahrenheit axis to temperature plot
        if ( strmatch(char(yAxisVariableName), ...
                      self.configuration.temperatureFieldName, 'exact') )
          set(axhan,'Box','off');
          yLimit2 = (ylim*9/5)+32;
          ax2 = axes('Position',get(axhan,'Position'),'Color','none','YaxisLocation','right');
          set(ax2,'YLim',yLimit2);
          set(ax2, ...
              'XColor'      , [.3 .3 .3], ...
              'YColor'      , [.3 .3 .3]  ...
          );
          ytick2=get(ax2,'ytick');
          yticklabel2 = num2str(ytick2');
          set(ax2,'XTick',nan,'YTick',ytick2);
          set(ax2,'XTickLabel',nan,'YTickLabel',yticklabel2);
          set(ax2,'XAxisLocation','top');
        end;
        
        % Labeling, Positions and Sizes of Objects
        axes(axhan);
        hylab = ylabel(figureYAxisLabel,'fontsize',10,'fontweight','normal'); 
        set(hylab,'units','normalized','position',yLabelPosition);
           % Label Fahrenheit axis on temperature plot
	         if ( strmatch(char(yAxisVariableName), ...
	                      self.configuration.temperatureFieldName, 'exact') )
               axes(ax2); 
               hylab2 = ylabel([yAxisVariableName ' (\circF)'],'fontsize',10,'fontweight','normal'); 
           end;
        set(figureHandle,'renderer','painters'); %,'visible','off');
      end
      value = figureHandle;
    end
    
   
    
   % A method used to create a time series figure for the PacIOOS website.
   % @returns value - handle of the figure object created
    function value = createPacIOOSFigures(             ...
                              self                   , ...
                              time                   , ...                              
                              figureTitlePrefix      , ...
                              figureDuration         , ...
                              yVariables             , ...
                              yAxisLabels            , ...
                              plotLocations          , ...
                              plotColors             , ...
                              variableRanges           ...
                                                            )
                                                        
                                                        
    if ( self.configuration.debug )
        disp('DataProcessor.createPacIOOSFigures() called.');
    end
    
    close(gcf);
    
   % Calculate start and end times, and the timestep to be used for
   % plotting the x axis
    
    if strcmp(figureDuration,'monthly')
        year=str2double(datestr(time(end),'yyyy'));
        month=str2double(datestr(time(end),'mm'));        
        startTime=datenum(year,month,1);
        if month ~= 12
            endTime=datenum(year,month+1,1);
        else
            endTime=datenum(year+1,1,1);
        end
        timeStep=5;
    else
        duration=str2double(figureDuration);
        endTime=ceil(now)+0.25;
        startTime=endTime-duration-0.25;
        timeStep=round(duration/7);
    end

    xTicks=(startTime:timeStep:endTime);
    
   % Create arrays for the axis properties
    numberofAxes=ceil(max(plotLocations)/2);
    baseAxisProperties=cell(numberofAxes,1);
    axisProperties=cell(length(yVariables),1);
    
    figureHandle=figure(); clf;

   % Set up the base axes
    for axisNumber = 1:numberofAxes
        subplot(4,1,axisNumber)
        baseAxisProperties{axisNumber}=gca;
        set(baseAxisProperties{axisNumber},                                 ...
                  'xlim',          [startTime endTime],                     ...
                  'xminorgrid',    'on',                                    ...
                  'xgrid',         'off',                                   ...
                  'ygrid',         'on',                                    ...
                  'xtick',         xTicks,                                  ...
                  'xticklabel',    datestr(xTicks,'mm/dd/yy'),              ...
                  'box',           'on',                                    ...
                  'position',      [0.13 0.96-0.21*axisNumber 0.7750 0.1321]...
                  )
  
  % Create the figure title
     if axisNumber == 1
      if strcmp(figureDuration,'monthly')
        date=datestr(time(end),'mmm yyyy');
        title({[figureTitlePrefix '        ' date];' '},        ...
               'fontweight','bold',                             ...
               'fontsize',11)
      else
        if time(end) > startTime
          startdate=datestr(startTime,'mm-dd-yyyy');
          enddate=datestr(time(end),'mm-dd-yyyy HH:MM');
          title({['\fontsize{11}' figureTitlePrefix];                       ...
                 '\fontsize{6} ';                                           ...
                 [ '\fontsize{10}' startdate '  ' 'to' '  ' enddate];' '},  ...
                 'fontweight','bold')
        else
          title({['\fontsize{11}' figureTitlePrefix];                       ...
                 '\fontsize{6} ';                                           ...
                 '\fontsize{10}No Data for this Period';' '},  ...
                 'fontweight','bold')
        end
      end
     end
     
    end
    
   % Plot the data
    for variableNumber=1:length(yVariables)
        
        yVariableName = char(yVariables{variableNumber});
        
        yDataLocation = find( ...
                         strcmp( ...
                          yVariableName, ...
                          self.configuration.dataVariableNames ...
                               ) ...
                              );
                          
   % Get the data for the y-axis  
        yVar = self.dataCellArray{yDataLocation};
   % Get the data units     
        yDataUnits = char(self.configuration.dataVariableUnits{yDataLocation});
   % Get the range for the current variable
        yRange = variableRanges{variableNumber};
   
   % Set the limits and ticks for the y-axis
        if strcmp(yRange{1},'fixed')
            minY = min(yRange{2});
            maxY = max(yRange{2});
            if length(yRange{2}) > 2
                if yRange{2}(3) > yRange{2}(2)
                    for n = 1:length(yRange{2})-1
                        if length(find(yVar>yRange{2}(n)))>30
                                maxY = yRange{2}(n+1);      
                        end
                    end
                else
                    for n = 1:length(yRange{2})-1
                        if length(find(yVar<yRange{2}(n)))>30
                                minY = yRange{2}(n+1);
                        end
                    end
                end
            end
        else
            if length(yRange) > 1
                multiplier = yRange{2}(1);
            else
                multiplier = 10;
            end
            minY = multiplier*floor(min(yVar)/multiplier);
            maxY = multiplier*ceil(max(yVar)/multiplier);
        end
        
   % Set the ticks for the y-axis     
        if length(yRange) == 3
            numberofTicks = yRange{3}(1);
        else
            numberofTicks = 5;
        end
        
        range = maxY - minY;
        
        yTicks = (minY:range/numberofTicks:maxY);
        
   % Create the tick labels for the y-axis     
        yTickLabels = cell(length(yTicks),1);
        
        if strcmp(yDataUnits,'%')
            for j=1:length(yTicks)
                yTickLabels{j}=strcat(num2str(yTicks(j)),'%');
            end
            tickLabelSize=9;
        else
            for j=1:length(yTicks)
                yTickLabels{j}=num2str(yTicks(j));
            end
            tickLabelSize=10;
        end
                
        
   % Get the location of the y-axis (left or right) for the current plot   
        if mod(plotLocations(variableNumber),2)
            yLocation='left';
        else
            yLocation='right';
        end
   
   % Get the properties of the base axis for the current plot     
        baseAxis=baseAxisProperties{ceil(plotLocations(variableNumber)/2)};
   % Set the axis properties for the current plot
        axisProperties{variableNumber} = axes(                         ...
                       'position',      get(baseAxis,'position'),  ...
                       'color',         'none',                    ...
                       'xtick',         xTicks,                    ...
                       'xlim',          [startTime endTime],       ...
                       'ylim',          [minY maxY],               ...
                       'ytick',         yTicks,                    ...
                       'ycolor',        plotColors{variableNumber},    ...
                       'xgrid',         'off',                     ...
                       'ygrid',         'off',                     ...
                       'xticklabel',    '',                        ...
                       'yaxislocation', yLocation,                 ...
                       'yticklabel',    yTickLabels,               ...
                       'fontsize',      tickLabelSize              ...
                       );
                   
   % Selectively remove single data points to prevent line from connecting
   % points more than 1/4 day apart
        for i=2:length(time)
           if (time(i)-time(i-1)) > 0.25
                yVar(i-1) = NaN;
           end
        end
   
   % Plot the current variable vs time               
        line(time,yVar,                                            ...
                       'color',         plotColors{variableNumber},    ...
                       'parent',        axisProperties{variableNumber} ...
                       )
   
   % Match the y-axis properties of the base axis to the current plot                   
        set(baseAxisProperties{ceil(plotLocations(variableNumber)/2)},                                 ...
                       'ylim',          get(axisProperties         ...
                                           {variableNumber},'ylim'),   ...
                       'ytick',         get(axisProperties         ...
                                           {variableNumber},'ytick'),  ...
                       'yticklabel',    ''                         ...
                  )

              
   % Label the y-axis           
        hylab = ylabel(yAxisLabels{variableNumber},                 ...
                       'fontsize',10,'fontweight','bold');
        set(hylab,'units','normalized');
        yLabelPosition = get(hylab,'position');
        if strcmp(yLocation,'left')
           yLabelPosition(1) = -0.065;
        else
           yLabelPosition(1) = 1.065;
        end
        set(hylab,'position',yLabelPosition);      
              
    end
    
    value = figureHandle;
    
    end %function
    
    
    
    
    % A method used to export a figure to various vector and raster-based image
    % formats.  This method requires ImageMagick to be installed on the processing
    % machine.
    % Todo: Needs work to make the export take format arguments
    % @param inputFigure - the figure to be exported
    % @param outputFormat - the desired raster or vector format (EPS, PNG, JPG, PDF)
    % @param figureNameSuffix - the suffix to append to the figure name
    % @returns void
    function export(self, inputFigure, outputFormat, figureNameSuffix)
     
      if ( self.configuration.debug )
        disp('DataProcessor.export() called.');
      end
          
          % Export to Enhanced Postscript
          timestamp = datestr(self.processTime, 'yyyymmddHHMMSS');
          fileNamePrefix = [self.configuration.rbnbSource  '_' ...
                            timestamp '_' ...
                            figureNameSuffix];
          try
            print(inputFigure,'-depsc2', ...
              [self.configuration.outputDirectory ...
               fileNamePrefix '.eps']);
          catch saveException
            disp(['There was an error saving the figure to EPS. ' ...
                  'The error message was:' saveException.message]  ...
                );
          end
             
          % Export to JPEG
          try
            print(inputFigure,'-djpeg100', ...
              [self.configuration.outputDirectory ...
               fileNamePrefix '.jpg']);
          catch saveException
            disp(['There was an error saving the figure to JPG. ' ...
                  'The error message was:' saveException.message]  ...
                );
          end
          % Export to JPG from the high resolution EPS file using a conversion
          % program as defined in the configuration instance
          
          % commented out the code to use ghostscript for JPG export because
          % of library incompatibility issues
          
          %if ( ~isempty(self.configuration.convertPath) && ...
          %     ~isempty(self.configuration.convertOptions) )
          %  try
          %    system(                                ...
          %      [                                    ...
          %        self.configuration.convertPath     ... % use the convert program
          %        self.configuration.convertOptions  ... % with convert options
          %        self.configuration.outputDirectory ... 
          %        fileNamePrefix '.jpg '             ... % convert to file name
          %        self.configuration.outputDirectory ... 
          %        fileNamePrefix '.eps'              ... % convert from file name                  
          %      ]                                    ...
          %    );
          %  catch exportException
          %    disp(['There was an error exporting the figure to JPG. ' ...
          %          'The error message was:' exportException.message]  ...
          %        );
          %  end
          %else
          %  disp(['Figure conversion to JPG skipped.  Please set the correct ' ...
          %        'values for the Configure.convertPath and the '              ...
          %        'Configure.convertOptions properties.']                      ...
          %      );
          %end
          
          % Copy the latest JPG to latest_{duration}day.jpg
          if ( ~isempty(self.configuration.copyPath) )
            
            % copy to latest JPG file
            try
              system(                                ...
                [                                    ...
                  self.configuration.copyPath ' '    ... % use the copy program
                  self.configuration.outputDirectory ... 
                  fileNamePrefix '.jpg '             ... % copy from file name
                  self.configuration.outputDirectory ... 
                  'latest_' figureNameSuffix '.jpg'  ... % copy to file name                  
                ]                                    ...
              );
            catch copyException
              disp(['There was an error copying the figure to latest_* JPG. ' ...
                    'The error message was:' copyException.message]           ...
                  );
            end
            
            % copy to latest EPS file
            try
              system(                                ...
                [                                    ...
                  self.configuration.copyPath ' '    ... % use the copy program
                  self.configuration.outputDirectory ... 
                  fileNamePrefix '.eps '             ... % copy from file name
                  self.configuration.outputDirectory ... 
                  'latest_' figureNameSuffix '.eps'  ... % copy to file name                  
                ]                                    ...
              );
            catch copyException
              disp(['There was an error copying the figure to latest_* EPS. ' ...
                    'The error message was:' copyException.message]           ...
                  );
            end
          else
            disp(['Figure copying to JPG skipped.  Please set the correct ' ...
                  'value for the Configure.copyPath property.']             ...
                );
          end
          
    end
    
    % A method used to export a figure to various vector and raster-based image
    % formats.  Used for exporting figures for the PacIOOS website.
    % @param inputFigure - the figure to be exported
    % @param outputFormat - the desired raster or vector format (EPS, PNG, JPG, PDF)
    % @param figureNameSuffix - the suffix to append to the figure name
    % @returns void
    function export2(                     ...
                      self              , ...
                      inputFigure       , ...
                      time              , ...
                      outputFormat      , ...
                      outDir            , ...
                      figureNameSuffix    ...
                                        )
     
      if ( self.configuration.debug )
        disp('DataProcessor.export2() called.');
      end
      
      %endTime=ceil(time(end));
      %startTime=endTime-figureDuration;
      year=datestr(now,'yyyy');
      month=datestr(now,'mm');
      %day=datestr(time(end),'dd');
      mkdirpath=self.configuration.mkdirPath;
      pathSeparator=self.configuration.pathSeparator;
      
      %check for existing directory and create a new one if necessary
      try
          if ~exist(outDir,'dir');
              system([mkdirpath ' ' outDir]);
          end
          if ~exist([outDir year],'dir');
              system([mkdirpath ' ' outDir year]);
          end
          if ~exist([outDir year pathSeparator month],'dir'); 
              system([mkdirpath ' ' outDir year pathSeparator month]);
          end
          %if ~exist([outDir year pathSeparator month ...
          %           pathSeparator day],'dir');
          %    system([mkdirpath ' ' outDir year ...
          %            pathSeparator month pathSeparator day]);
          %end
      catch dirException
          disp (['There was an error finding or creating the output directory. ' ...
                 'The error message was:' dirException.message]);
      end
          
      outDirectory=[outDir year pathSeparator                    ...
                    month pathSeparator];   % day pathSeparator];
      
      % Export to Enhanced Postscript
      fileNamePrefix = [self.configuration.rbnbSource  '_'       ...
                    ...   %datestr(startTime,'yyyymmddHHMM') '-' ...
                        datestr(now,'yyyymmdd') '_'              ...
                        figureNameSuffix];
                        
      % Export to eps
      if sum(strcmp(outputFormat,'.eps'))  
            try
                print(inputFigure,'-depsc2',                           ...
                [outDirectory fileNamePrefix '.eps']);
            catch saveException
                disp(['There was an error saving the figure to EPS. '  ...
                      'The error message was:' saveException.message]  ...
                    );
            end
      end
             
      % Export to JPEG
      if sum(strcmp(outputFormat,'.jpg'))
          try
            print(inputFigure,'-djpeg100',                         ...
              [outDirectory fileNamePrefix '.jpg']);
          catch saveException
            disp(['There was an error saving the figure to JPG. '  ...
                  'The error message was:' saveException.message]  ...
                );
          end
      end

          
      % Copy the latest JPG to latest_{duration}day.jpg
      if ( ~isempty(self.configuration.copyPath) )
            
      % copy to latest JPG file
      if sum(strcmp(outputFormat,'.jpg'))
            try
              system(                                ...
                [                                    ...
                  self.configuration.copyPath ' '    ... % use the copy program
                  outDirectory                       ... 
                  fileNamePrefix '.jpg ' ' '         ... % copy from file name
                  outDir                             ... 
                  'latest_' figureNameSuffix '.jpg'  ... % copy to file name                  
                ]                                    ...
              );
            catch copyException
              disp(['There was an error copying the figure to latest_* JPG. ' ...
                    'The error message was:' copyException.message]           ...
                  );
            end
      end
            
      % copy to latest EPS file
      if sum(strcmp(outputFormat,'.eps'))
            try
              system(                                ...
                [                                    ...
                  self.configuration.copyPath ' '    ... % use the copy program
                  outDirectory                       ... 
                  fileNamePrefix '.eps ' ' '         ... % copy from file name
                  outDir                             ... 
                  'latest_' figureNameSuffix '.eps'  ... % copy to file name                  
                ]                                    ...
              );
            catch copyException
              disp(['There was an error copying the figure to latest_* EPS. ' ...
                    'The error message was:' copyException.message]           ...
                  );
            end
      end
     end
          
    end %function
    
    
    % A method used to export a figure to various vector and raster-based image
    % formats.  Used specifically for exporting monthly plots for the PacIOOS website.
    % @param inputFigure - the figure to be exported
    % @param outputFormat - the desired raster or vector format (EPS, PNG, JPG, PDF)
    % @param figureNameSuffix - the suffix to append to the figure name
    % @returns void 
    
    function exportMonthly(               ...
                      self              , ...
                      inputFigure       , ...
                      time              , ...
                      outputFormat      , ...
                      outDir            , ...
                      figureNameSuffix    ...
                                        )
     
      if ( self.configuration.debug )
        disp('DataProcessor.exportMonthly() called.');
      end
      
      year=datestr(time(end),'yyyy');
      mkdirpath=self.configuration.mkdirPath;
      pathSeparator=self.configuration.pathSeparator;
      
      %check for existing directory and create a new one if necessary
      try
          if ~exist(outDir,'dir');
              system([mkdirpath ' ' outDir]);
          end
          if ~exist([outDir pathSeparator year],'dir');
              system([mkdirpath ' ' outDir pathSeparator year]);
          end

      catch dirException
          disp (['There was an error finding or creating the output directory. ' ...
                 'The error message was:' dirException.message]);
      end
      
      % Define Output Directory
      outDirectory=[outDir pathSeparator ...
                    year pathSeparator];
      
      % Export to Enhanced Postscript
      fileNamePrefix = [self.configuration.rbnbSource  '_' ...
                        figureNameSuffix];
                        
      % Export to eps
      if sum(strcmp(outputFormat,'.eps'))  
            try
                print(inputFigure,'-depsc2',                           ...
                [outDirectory fileNamePrefix '.eps']);
            catch saveException
                disp(['There was an error saving the figure to EPS. '  ...
                      'The error message was:' saveException.message]  ...
                    );
            end
      end
             
      % Export to JPEG
      if sum(strcmp(outputFormat,'.jpg'))
          try
            print(inputFigure,'-djpeg100',                         ...
              [outDirectory fileNamePrefix '.jpg']);
          catch saveException
            disp(['There was an error saving the figure to JPG. '  ...
                  'The error message was:' saveException.message]  ...
                );
          end
      end
          
    end %function
    
    % A method used to fetch the ASCII data string for the given RBNB
    % Data Turbine source, channel, reference, and given time duration
    % @todo - support the RBNB 'absolute' reference
    % @param source - the name of the RBNB Data source instrument
    % @param channel -  the name of the RBNB Data channel
    % @param reference - the reference datum for the time series (newest, oldest)
    % @param duration - the duration of the time series to process in seconds
    function [dataString, dataTimes, dataName] = getRBNBData(self)
      
      if ( self.configuration.debug )
        disp('DataProcessor.getRBNBData() called.');
      end
      
      % set the pertinent properties of this DataProcessor object
      
      % Create a new sink client to the DataTurbine
      try
      matlabSink = rbnb_sink( ...
        [self.configuration.rbnbServer ':' ...
         self.configuration.rbnbPort], ...
         self.configuration.rbnbSinkName);
      catch rbnbSinkException
        disp('Could not create RBNB sink client.');
        disp(rbnbSinkException.message);
      end
      
      % define the request details (get the latest 7 days of data)
      fullChannelName = [self.configuration.rbnbSource '/' self.configuration.rbnbChannel];
      
      % make the request to the DataTurbine and close the connection
      try
      
        if ( ~isempty(self.configuration.dataStartDate) && ...
             (strcmp(self.configuration.reference, 'absolute')) )
          secondsToTheEpoch  = datenum('01-01-1970 00:00:00') * 24 * 60 * 60;
          secondsToStartDate = datenum(self.configuration.dataStartDate) * 24 * 60 * 60;                
          self.configuration.startTime = secondsToStartDate - secondsToTheEpoch;
                    
        end
        
        [dataString, ...
         dataTimes,  ...
         dataName] = ...
          rbnb_get(matlabSink, fullChannelName, self.configuration.startTime, ...
          self.configuration.duration, self.configuration.reference);
        matlabSink.CloseRBNBConnection;
        clear matlabSink fullChannelName;
        
        % write the data to disk  
        % fd = fopen([self.configuration.rbnbSource '.10.1.dat'], 'w', 'l');
        % fwrite(fd, mostRecentData, 'int8');
        % fclose(fd);
      
      catch rbnbChannelException
        disp('Could not get channel data.  Setting values to null.');
        disp(rbnbChannelException.message);
        dataString = [];
        dataTimes = [];
        dataName = '';
      end
    end % getRBNBData
    
    % A method used to fetch the ASCII data string from the given RBNB
    % archive, for use when the Data Turbine channel is incomplete
    function [dataString, dataTimes, dataName] = getArchiveData(self)
        
      if ( self.configuration.debug )
        disp('DataProcessor.getArchiveData() called.');
      end

      %IF reading from archive; be sure to change 'duration' to 2678400 (31 days) 
      %Grab the data for 31 days - saved in data_'rbnbSource'.dat in read_archive folder
      read_archiveSourceFile=[self.configuration.read_archivePath       ...
                'read_archive_' self.configuration.rbnbSource '.sh'];
      
      % Create a new read_archive source file from a template (if necessary)
      if ~exist(read_archiveSourceFile,'file') || self.configuration.duration_days  ...
              > 32 || ~strcmp(self.configuration.dataEndDate, ' ')
          
          if ~strcmp(self.configuration.dataEndDate, ' ')
               endDate = datestr(datenum(self.configuration.dataEndDate, ...
                           'mm-dd-yyyy HH:MM:SS'),'yyyymmdd');
          else
               endDate = '+2 days';
          end
          
         read_archiveFile=[self.configuration.read_archivePath 'read_archive.sh'];
         eval(['!sed -e "s/REPLACENAME/"' self.configuration.rbnbSource '/ '        ...
                read_archiveFile ' > ' [read_archiveSourceFile '.tmp']]);
         eval(['!sed -e "s/REPLACEFIELD/"' self.configuration.archiveDirectory '/ ' ...
               [read_archiveSourceFile '.tmp'] ' > ' [read_archiveSourceFile '.tmp2']]);
         eval(['!sed -e "s/REPLACEENDDAY/""' endDate '"/ ' ...
               [read_archiveSourceFile '.tmp2'] ' > ' [read_archiveSourceFile '.tmp3']]);
         eval(['!sed -e "s/REPLACEDURATION/"' num2str(self.configuration.duration_days) ...
               '/ ' [read_archiveSourceFile '.tmp3'] ' > ' read_archiveSourceFile]);  
         system(['rm' ' ' [read_archiveSourceFile '.tmp']]);
         system(['rm' ' ' [read_archiveSourceFile '.tmp2']]);
         system(['rm' ' ' [read_archiveSourceFile '.tmp3']]);
      end
      eval(['!sh ' read_archiveSourceFile]);
        
	  %read in file reading from archive
	  fid=fopen([self.configuration.read_archivePath 'data_'       ...
                  self.configuration.rbnbSource '.dat']); 
	  tline=fgets(fid);
      while ischar(tline);
          if length(tline)>40
            dataString(1,:)=tline;
            dataTimes(1,:)=now;%(tline(end-20:end));
            break
          end
          tline=fgets(fid);
      end
	  i=2;
      stringLength=length(dataString);
      while ischar(tline); 
		tline=fgets(fid);
        if length(tline)>40
     % Pad string or array with blank spacing to account for length differences   
          if length(tline)<stringLength
                dsSize=size(dataString);
                tline=[tline,blanks(dsSize(2)-length(tline))];
          else if length(tline)>stringLength
                dsSize=size(dataString);
                pads=repmat(blanks(length(tline)-dsSize(2)),dsSize(1),1);
                dataString=[dataString pads];
                stringLength=length(tline);
               end
          end
        dataString(i,:)=tline;
        i=i+1;
        end
      end
      fclose(fid);
      clear fid i tline;

      dataName =  [self.configuration.rbnbSource '/' self.configuration.rbnbChannel];
       
    end %getArchiveData
    
    % A method used to fetch ASCII data from the LOBOViz database
    function [dataString, dataTimes, dataName] = getLOBOVizData(self)

     % Set the reference dates 
     min_date=datestr(now - self.configuration.duration_days, 'yyyymmdd');
     if ~strcmp(self.configuration.dataEndDate, ' ')
         max_date=datestr(datenum(self.configuration.dataEndDate, ...
             'mm-dd-yyyy HH:MM:SS'),'yyyymmdd');
     else
         max_date=datestr(now, 'yyyymmdd');
     end
     
     % Create the URL string
     LOBOVizSource=['"' self.configuration.LOBOVizURL ...
                    '&min_date=' min_date '&max_date=' max_date '"'];
    
     % Get the file path for temporary local data storage
     LOBOVizFilePath=[self.configuration.LOBOVizPath self.configuration.rbnbSource];
    
     % Retrieve the last 32 days of data from the LOBOViz database 
     % and write to a local directory 
     try
      system([self.configuration.wgetPath ' ' LOBOVizSource ...
              ' --output-document=' LOBOVizFilePath]);     
     catch dataException
          disp (['There was an error retrieving data from the LOBOViz Database. ' ...
                 'The error message was:' dataException.message]);
     end
    
     % Read the data from the local file into a string array 
     fid=fopen(LOBOVizFilePath);
     tline=fgets(fid);
     while ischar(tline);
      if str2num(tline(1))
         if length(tline)>40
            dataString(1,:)=tline;
            dataTimes(1,:)=now;
            break
         end
      end
      tline=fgets(fid);
     end
     i=2;
      stringLength=length(dataString);
      while ischar(tline); 
		tline=fgets(fid);
        if length(tline)>40
     % Pad string or array with blank spacing to account for length differences   
          if length(tline)<stringLength
                dsSize=size(dataString);
                tline=[tline,blanks(dsSize(2)-length(tline))];
          else if length(tline)>stringLength
                dsSize=size(dataString);
                pads=repmat(blanks(length(tline)-dsSize(2)),dsSize(1),1);
                dataString=[dataString pads];
                stringLength=length(tline);
               end
          end
        dataString(i,:)=tline;
        i=i+1;
        end
      end
      fclose(fid);
      clear fid i tline;

      dataName =  [self.configuration.rbnbSource '/' self.configuration.rbnbChannel];
end  %getLOBOVizData
    
    % A method that appends a new variable field name to the dataVariableNames
    % property in order to keep track of variable names inside of the 
    % dataCellArray
    function updateDataVariableNames(self, newFieldName)
      
      if ( self.configuration.debug )
        disp('DataProcessor.updateDataVariableNames() called.');
      end
      
      % if the variable name does not already exist
      if ( ~any( ...
             strcmp( ...
               newFieldName, ...
               self.configuration.dataVariableNames ...
             ) ...
           ) ...
         )
         
        % add the new data column into the variable name and unit properties
        config = get(self, 'configuration');
        variableNameCellArray = get(config, 'dataVariableNames');      
        variableNameCellArray{length(variableNameCellArray) + 1} = newFieldName;      
        set(config, 'dataVariableNames', variableNameCellArray);     
        clear config variableNameCellArray;
     % else
     % disp('Error: Variable already exists!')
      end
    end
    
    % A method that appends a new variable unit to the dataVariableUnits
    % property in order to keep track of variable units inside of the 
    % dataCellArray
    function updateDataVariableUnits(self, newUnitName)
      
      if ( self.configuration.debug )
        disp('DataProcessor.updateDataVariableUnits() called.');
      end
      
      if length(self.configuration.dataVariableNames) > ...
              length(self.configuration.dataVariableUnits)
        
        % add the new data column into the variable name and unit properties
        config = get(self, 'configuration');
        variableUnitCellArray = get(config, 'dataVariableUnits');           
        variableUnitCellArray{length(variableUnitCellArray) + 1} = newUnitName;      
        set(config, 'dataVariableUnits', variableUnitCellArray);      
        clear config variableUnitCellArray;
      end
    end
    
    % --------------%
    % Setter methods 
    % --------------%

    % A setter method for the dataString property
    function self = set.dataString(self, value) 
      self.dataString = char(value)';  
    end
    
    % A setter method for the timerStartTime property
    function self = set.timerStartTime(self, interval)
      % calculate the next start time based on the timer interval 
      value = ceil(now*24*60/interval)/(24*60/interval) ...
              + interval/60/24;
      self.timerStartTime = value;  
    end

    % --------------%
    % Getter methods 
    % --------------%
    
  end % methods
  
  events
  end % events
end % class definition
