package edu.ucsb.nceas.metacat.client.gsi;

import edu.ucsb.nceas.utilities.HttpMessage;
import org.ietf.jgss.GSSCredential;

import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.net.URLConnection;
import java.util.Properties;
import java.lang.reflect.Method;
import java.lang.reflect.InvocationTargetException;

/** An extension of {@link HttpMessage} that uses Grid Security Infrastructure
 *  (GSI) enabled HTTPS instead of HTTP to communicate.
 *
 *  <p>Note that not all client deployments will include the JARs necessary to
 *  run this version of the Metacat client; therefore, we should make sure that
 *  the superclass (MetacatClient) can run even if this class can't be loaded.
 *  That is, catch (and log) NoClassDefFoundError, etc. */
public class HttpGsiMessage extends HttpMessage {
	private GSSCredential credential;

	public HttpGsiMessage(GSSCredential credential, URL servlet) {
		super(servlet); // init our URL
		// GSI SSL output stream can't handle flushes until it's closed,
		// or else it terminates the connection
		ignoreOutputStreamFlushes = true;
		if (credential == null)
			throw new NullPointerException("credential is null.");
		this.credential = credential;
	}

	/**
	 * Performs a GET request to the previously given servlet
	 * Builds a query string from the supplied Properties list.
	 */
	public InputStream sendGetMessage(Properties args) throws IOException {
		String argString = "";//default
		if (args != null) argString = "?" + toEncodedString(args);
		URLConnection urlCon
			= initConnection(servlet.toExternalForm() + argString);
		return urlCon.getInputStream();
	}

	protected void openPostConnection() throws IOException {
		super.openPostConnection();
		initConnection(con);
	}

	private URLConnection initConnection(String urlString)
		throws IOException
	{
		URL url = new URL(urlString);
		// expect to actually be an org.globus.net.GSIHttpURLConnection
		URLConnection gsiCon = url.openConnection();
		return initConnection(gsiCon);
	}

	private void checkClass(Object o, String className) {
		if (!o.getClass().getName().equals(className))
			throw new IllegalStateException("Expected a " + className
				+ " but received a " + o.getClass().getName() + ".  "
				+ "Possible causes: cog-jglobus.jar may not be available, or "
				+ "MetacatGSIClient.register() may not have been called, or "
				+ "it may have been called after Java's protocol handlers "
				+ "were already initialized -- try calling it earlier.");
	}

	public static final String GSI_HTTP_URL_CONN_CLASSNAME
		= "org.globus.net.GSIHttpURLConnection";

	public static final String GSI_HTTPS_HANDLER_CLASSNAME
		= "org.globus.net.protocol.https.Handler";

	/** Initialize a GSI HTTP connection.
	 *  @param gsiCon expected to be an org.globus.net.GSIHttpURLConnection,
	 *  but to avoid requiring people to use cog-jglobus.jar, let's make method
	 *  calls via reflection.
	 *  @return actually an org.globus.net.GSIHttpURLConnection */
	private URLConnection initConnection(URLConnection gsiCon) {
		// set GSI credentials
		checkClass(gsiCon, GSI_HTTP_URL_CONN_CLASSNAME);
		try {
			Method setCredentials = gsiCon.getClass().getMethod
				("setCredentials", new Class[] { GSSCredential.class });
			setCredentials.invoke(gsiCon, new Object[] { credential });
		} catch (NoSuchMethodException e) {
			IllegalStateException ise = new IllegalStateException
				("Expected to find and invoke " + GSI_HTTP_URL_CONN_CLASSNAME
					+ ".setCredentials(GSSCredential).");
			ise.initCause(e);
			throw ise;
		} catch (IllegalAccessException e) {
			IllegalStateException ise = new IllegalStateException
				("Expected to find and invoke " + GSI_HTTP_URL_CONN_CLASSNAME
					+ ".setCredentials(GSSCredential).");
			ise.initCause(e);
			throw ise;
		} catch (InvocationTargetException e) {
			if (e.getCause() instanceof RuntimeException)
				throw (RuntimeException) e.getCause();
			if (e.getCause() instanceof Error)
				throw (Error) e.getCause();
			else {
				IllegalStateException ise = new IllegalStateException
					("Unexpected checked exception.");
				ise.initCause(e.getCause());
				throw ise;
			}
		}

		// the one line that we replaced with that enormous block above
		// set GSI credentials
		// ((GSIHttpURLConnection) gsiCon).setCredentials(credential);

		// turn off caching
		con = gsiCon;
		con.setUseCaches(false);

		return gsiCon;
	}
}
