/*
 * Copyright (c) 2009, Dennis M. Sosnoski. All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without modification, are permitted provided that the
 * following conditions are met:
 * 
 * Redistributions of source code must retain the above copyright notice, this list of conditions and the following
 * disclaimer. Redistributions in binary form must reproduce the above copyright notice, this list of conditions and the
 * following disclaimer in the documentation and/or other materials provided with the distribution. Neither the name of
 * JiBX nor the names of its contributors may be used to endorse or promote products derived from this software without
 * specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package org.dataone.jibx.schema.codegen.extend;

import java.util.Iterator;
import org.eclipse.jdt.core.dom.AST;
import org.eclipse.jdt.core.dom.ASTNode;
import org.eclipse.jdt.core.dom.ASTParser;
import org.eclipse.jdt.core.dom.CompilationUnit;
import org.eclipse.jdt.core.dom.FieldDeclaration;
import org.eclipse.jdt.core.dom.MethodDeclaration;
import org.eclipse.jdt.core.dom.TypeDeclaration;
import org.eclipse.jdt.core.dom.VariableDeclarationFragment;
import org.jibx.binding.model.ElementBase;
import org.jibx.schema.codegen.IClassHolder;
import org.jibx.schema.codegen.NameUtils;
import org.jibx.schema.codegen.extend.ClassDecorator;
import org.jibx.schema.codegen.extend.NameMatchDecoratorBase;
import org.jibx.util.NameUtilities;

/**
 * Code generation decorator which adds the <code>java.lang.Serializable</code> interface to each class, and optionally
 * also adds a <code>private static final long serialVersionUID</code> value.
 */
public class ComparableIdentifierDecorator extends NameMatchDecoratorBase implements ClassDecorator
{
   // where $0 is the description text, $1 is the field name, $2 is the value name with initial uppercase character,
   //  $3 is the type, and $4 is a cast if an untyped list is used, or empty if a typed list is used
    private static final String s_classText = "class Gorph { " +
        "/**  1 $1 2 $2 3 $3 4 $4 5 $5  value is a string, override equals of $5.\n" +
        "* @param other\n" +
        "* @return boolean\n */\n" +
        "@Override  public boolean equals(Object other) { if ( other == null || other.getClass() != this.getClass() ) return false; $5 other$5 = ($5)other;  return $1.equals(other$5.getValue()); }" +
        "/** return the hashcode of $5's string value.\n"
      + " * @return int\n"
      + " */\n"
      + "@Override public int hashCode() { return $1.hashCode(); }\n" +
        "/** Compares order based on the String value of two objects of the same type.\n"
      + " * @param other\n"
      + " * @return int\n"
      + " * @throws ClassCastException */\n"
      + "@Override public int compareTo(Object other) throws ClassCastException {$5 other$5 = ($5)other; return $1.compareTo(other$5.getValue()); }";

    
    /** Parser instance used by class. */
    private final ASTParser m_parser = ASTParser.newParser(AST.JLS3);
    
    /** Serial version UID value (<code>null</code> if not set). */
    private Long m_serialVersion;
    /**
     * Replace all occurrences of one string with another in a buffer.
     *
     * @param match
     * @param replace
     * @param buff
     */
    private static void replace(String match, String replace, StringBuffer buff) {
        int base = 0;
        while ((base = buff.indexOf(match, base)) >= 0) {
            buff.replace(base, base+match.length(), replace);
        }
    }
    /**
     * Set serial version.
     *
     * @param version
     */
    public void setSerialVersion(Long version) {
        m_serialVersion = version;
    }
    
    /**
     * Method called after completing code generation for the target class.
     *
     * @param binding 
     * @param holder
     */
    public void finish(ElementBase binding, IClassHolder holder) {}
    
    /**
     * Method called before starting code generation for the target class.
     *
     * @param holder
     */
    public void start(IClassHolder holder) {

    }
    
    /**
     * Method called after adding each data value to class.
     *
     * @param basename base name used for data value
     * @param collect repeated value flag
     * @param type value type (item value type, in the case of a repeated value)
     * @param field actual field
     * @param getmeth read access method (<code>null</code> if a flag value)
     * @param setmeth write access method (<code>null</code> if a flag value)
     * @param descript value description text
     * @param holder
     */
    public void valueAdded(String basename, boolean collect, String type, FieldDeclaration field,
        MethodDeclaration getmeth, MethodDeclaration setmeth, String descript, IClassHolder holder) {
        String fieldtype = field.getType().toString();
        if (matchName(holder.getName())) {
            holder.addInterface("Comparable");
        }

        if (matchName(holder.getName())) {

            // make substitutions in template text
            StringBuffer buff = new StringBuffer(s_classText);
            VariableDeclarationFragment vardecl = (VariableDeclarationFragment)field.fragments().get(0);
            replace("$0", descript, buff);
            replace("$1", vardecl.getName().getIdentifier(), buff);
            replace("$2", NameUtilities.depluralize(NameUtils.toNameWord(basename)), buff);
            replace("$3", holder.getTypeName(type), buff);
            String cast = field.getType().isParameterizedType() ? "" : ("(" + type + ")");
            replace("$4", cast, buff);
            replace("$5",holder.getName(), buff);

            // parse the resulting text
            m_parser.setSource(buff.toString().toCharArray());
            CompilationUnit unit = (CompilationUnit)m_parser.createAST(null);

            // add all methods from output tree to class under construction
            TypeDeclaration typedecl = (TypeDeclaration)unit.types().get(0);
            for (Iterator iter = typedecl.bodyDeclarations().iterator(); iter.hasNext();) {
                ASTNode node = (ASTNode)iter.next();
                if (node instanceof MethodDeclaration) {
                    holder.addMethod((MethodDeclaration)node);
                }
            }

        }
    }
}
