/*
 * Decompiled with CFR 0.152.
 */
package org.dataone.bookkeeper.resources;

import com.codahale.metrics.annotation.Timed;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.ObjectNode;
import io.dropwizard.jackson.Jackson;
import java.time.Instant;
import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Set;
import java.util.UUID;
import javax.annotation.security.PermitAll;
import javax.validation.Valid;
import javax.validation.constraints.NotNull;
import javax.ws.rs.Consumes;
import javax.ws.rs.DELETE;
import javax.ws.rs.DefaultValue;
import javax.ws.rs.GET;
import javax.ws.rs.POST;
import javax.ws.rs.PUT;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.QueryParam;
import javax.ws.rs.WebApplicationException;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.Response;
import javax.ws.rs.core.SecurityContext;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.dataone.bookkeeper.api.Customer;
import org.dataone.bookkeeper.api.Feature;
import org.dataone.bookkeeper.api.Order;
import org.dataone.bookkeeper.api.OrderItem;
import org.dataone.bookkeeper.api.OrderList;
import org.dataone.bookkeeper.api.Product;
import org.dataone.bookkeeper.api.Quota;
import org.dataone.bookkeeper.jdbi.CustomerStore;
import org.dataone.bookkeeper.jdbi.OrderStore;
import org.dataone.bookkeeper.jdbi.ProductStore;
import org.dataone.bookkeeper.jdbi.QuotaStore;
import org.dataone.bookkeeper.resources.BaseResource;
import org.dataone.bookkeeper.security.DataONEAuthHelper;
import org.jdbi.v3.core.Jdbi;

@Timed
@Path(value="/orders")
@Produces(value={"application/json"})
public class OrdersResource
extends BaseResource {
    private final Log log = LogFactory.getLog(OrdersResource.class);
    private final OrderStore orderStore;
    private final ProductStore productStore;
    private final QuotaStore quotaStore;
    private final CustomerStore customerStore;
    private final ObjectMapper mapper = Jackson.newObjectMapper();
    private final DataONEAuthHelper dataoneAuthHelper;

    public OrdersResource(Jdbi database, DataONEAuthHelper dataoneAuthHelper) {
        this.orderStore = (OrderStore)database.onDemand(OrderStore.class);
        this.productStore = (ProductStore)database.onDemand(ProductStore.class);
        this.quotaStore = (QuotaStore)database.onDemand(QuotaStore.class);
        this.customerStore = (CustomerStore)database.onDemand(CustomerStore.class);
        this.dataoneAuthHelper = dataoneAuthHelper;
    }

    @Timed
    @GET
    @PermitAll
    @Produces(value={"application/json"})
    public OrderList listOrders(@Context SecurityContext context, @QueryParam(value="start") @DefaultValue(value="0") Integer start, @QueryParam(value="count") @DefaultValue(value="1000") Integer count, @QueryParam(value="subject") Set<String> subjects, @QueryParam(value="customerId") Integer customerId) throws WebApplicationException {
        Customer caller = (Customer)context.getUserPrincipal();
        boolean isAdmin = this.dataoneAuthHelper.isAdmin(caller.getSubject());
        List<Object> orders = new ArrayList();
        ArrayList<String> associatedSubjects = new ArrayList<String>();
        try {
            if (isAdmin) {
                if (customerId != null) {
                    orders = this.orderStore.findOrdersByCustomerId(customerId);
                } else if (subjects != null && !subjects.isEmpty()) {
                    associatedSubjects.addAll(subjects);
                    orders = this.orderStore.findOrdersBySubjects(associatedSubjects);
                } else {
                    orders = this.orderStore.listOrders();
                }
            } else if (customerId != null) {
                Customer existing = this.customerStore.getCustomer(customerId);
                if (!caller.getSubject().equals(existing.getSubject())) {
                    throw new WebApplicationException("Caller doesn't have access to this record.", Response.Status.FORBIDDEN);
                }
                orders = this.orderStore.findOrdersByCustomerId(customerId);
            } else if (subjects != null && !subjects.isEmpty()) {
                Set<String> desiredSubjects = this.dataoneAuthHelper.filterByAssociatedSubjects(caller, subjects);
                associatedSubjects.addAll(desiredSubjects);
                List<Order> associatedOrders = this.orderStore.findOrdersBySubjects(associatedSubjects);
                for (Order order : associatedOrders) {
                    Customer existing = this.customerStore.findCustomerBySubject(order.getSubject());
                    if (!caller.getSubject().equals(existing.getSubject())) {
                        Order redactedOrder = new Order();
                        redactedOrder.setId(order.getId());
                        redactedOrder.setSeriesId(order.getSeriesId());
                        redactedOrder.setObject(order.getObject());
                        redactedOrder.setName(order.getName());
                        redactedOrder.setStatus(order.getStatus());
                        redactedOrder.setCreated(order.getCreated());
                        redactedOrder.setUpdated(order.getUpdated());
                        redactedOrder.setEndDate(order.getEndDate());
                        redactedOrder.setStartDate(order.getStartDate());
                        redactedOrder.setQuotas(order.getQuotas());
                        orders.add(redactedOrder);
                        continue;
                    }
                    orders.add(order);
                }
            }
        }
        catch (Exception e) {
            e.printStackTrace();
            String message = "Couldn't list the orders due to an internal error.";
            throw new WebApplicationException(message, Response.Status.INTERNAL_SERVER_ERROR);
        }
        if (orders.isEmpty()) {
            throw new WebApplicationException("No orders were found.", Response.Status.NOT_FOUND);
        }
        return new OrderList(orders);
    }

    @Timed
    @POST
    @PermitAll
    @Consumes(value={"application/json"})
    public Order create(@Context SecurityContext context, @NotNull @Valid Order order) throws WebApplicationException {
        ObjectMapper mapper = Jackson.newObjectMapper();
        Customer caller = (Customer)context.getUserPrincipal();
        boolean isAdmin = this.dataoneAuthHelper.isAdmin(caller.getSubject());
        int now = new Integer((int)Instant.now().getEpochSecond());
        try {
            order.setStatus("created");
            if (order.getSeriesId() == null) {
                order.setSeriesId("urn:uuid:".concat(UUID.randomUUID().toString()));
            }
            ObjectNode statusTransitions = mapper.createObjectNode();
            statusTransitions.put("created", now);
            order.setStatusTransitions(statusTransitions);
            order.setCreated(Integer.valueOf(now));
            Product product = null;
            for (OrderItem item : order.getItems()) {
                product = this.productStore.getProduct(item.getParent());
                if (product == null) {
                    String message = "Couldn't find parent product for order item.";
                    throw new WebApplicationException(message, Response.Status.NOT_FOUND);
                }
                if (item.getType().equals("sku")) {
                    item.setAmount(product.getAmount());
                }
                item.setCurrency("USD");
                item.setDescription(product.getStatementDescriptor());
            }
            order.setAmount(order.getTotalAmount());
            if (!isAdmin) {
                order.setCustomer(caller.getId());
            }
            Integer id = this.orderStore.insert(order);
            order = this.orderStore.getOrder(id);
        }
        catch (Exception e) {
            String message = "Couldn't insert the order: " + e.getMessage();
            throw new WebApplicationException(message, Response.Status.INTERNAL_SERVER_ERROR);
        }
        return order;
    }

    @Timed
    @GET
    @PermitAll
    @Produces(value={"application/json"})
    @Path(value="{orderId}")
    public Order retrieve(@Context SecurityContext context, @PathParam(value="orderId") @NotNull Integer orderId) throws WebApplicationException {
        Customer caller = (Customer)context.getUserPrincipal();
        boolean isAdmin = this.dataoneAuthHelper.isAdmin(caller.getSubject());
        Order order = null;
        try {
            order = this.orderStore.getOrder(orderId);
            if (!isAdmin && !order.getCustomer().equals(caller.getId())) {
                throw new Exception("Customer doesn't have access to this order.");
            }
        }
        catch (Exception e) {
            String message = "Couldn't get the order: " + e.getMessage();
            throw new WebApplicationException(message, Response.Status.NOT_FOUND);
        }
        return order;
    }

    @Timed
    @PUT
    @PermitAll
    @Produces(value={"application/json"})
    @Path(value="{orderId}")
    public Order update(@Context SecurityContext context, @NotNull @Valid Order order) throws WebApplicationException {
        Customer caller = (Customer)context.getUserPrincipal();
        boolean isAdmin = this.dataoneAuthHelper.isAdmin(caller.getSubject());
        Order existing = this.orderStore.getOrder(order.getId());
        if (existing == null) {
            String message = "Couldn't find the order with id " + order.getId();
            throw new WebApplicationException(message, Response.Status.NOT_FOUND);
        }
        if (!isAdmin && !existing.getCustomer().equals(caller.getId())) {
            throw new WebApplicationException("Customer doesn't have access to this order.", Response.Status.UNAUTHORIZED);
        }
        try {
            order.setSeriesId(existing.getSeriesId());
            order.setCreated(existing.getCreated());
            order.setCurrency(existing.getCurrency());
            if (existing.getEmail() != null) {
                order.setEmail(existing.getEmail());
            }
            order.setCustomer(existing.getCustomer());
            if (existing.getStatusTransitions() != null) {
                order.setStatusTransitions(existing.getStatusTransitions());
            }
            order.setStatus("created");
            order.setUpdated(new Integer((int)Instant.now().getEpochSecond()));
            order.setMetadata(existing.getMetadata());
            order.setStartDate(existing.getStartDate());
            order.setEndDate(existing.getEndDate());
            order.setCharge(existing.getCharge());
            order.setAmountReturned(existing.getAmountReturned());
            order.setCurrency(existing.getCurrency());
            Product product = null;
            for (OrderItem item : order.getItems()) {
                product = this.productStore.getProduct(item.getParent());
                if (product == null) {
                    String message = "Couldn't find parent product for order item.";
                    throw new WebApplicationException(message, Response.Status.NOT_FOUND);
                }
                if (item.getType().equals("sku")) {
                    item.setAmount(product.getAmount());
                }
                item.setCurrency("USD");
                item.setDescription(product.getStatementDescriptor());
            }
            order.setAmount(order.getTotalAmount());
            this.orderStore.update(order);
        }
        catch (Exception e) {
            String message = "Couldn't update the order: " + e.getMessage();
            throw new WebApplicationException(message, Response.Status.INTERNAL_SERVER_ERROR);
        }
        return order;
    }

    @Timed
    @POST
    @PermitAll
    @Produces(value={"application/json"})
    @Path(value="{orderId}/pay")
    public Order pay(@Context SecurityContext context, @NotNull @PathParam(value="orderId") Integer orderId) throws WebApplicationException {
        Order order = null;
        Integer secondsSinceEpoch = new Integer((int)Instant.now().getEpochSecond());
        Customer caller = (Customer)context.getUserPrincipal();
        boolean isAdmin = this.dataoneAuthHelper.isAdmin(caller.getSubject());
        long trialDurationDays = this.dataoneAuthHelper.getConfiguration().getTrialDurationDays();
        Integer trialEndSecondsSinceEpoch = new Integer((int)Instant.ofEpochSecond(secondsSinceEpoch.intValue()).plus(trialDurationDays, ChronoUnit.DAYS).getEpochSecond());
        try {
            order = this.orderStore.getOrder(orderId);
            if (!isAdmin && !order.getCustomer().equals(caller.getId())) {
                throw new WebApplicationException("Customer doesn't have access to this order.", Response.Status.UNAUTHORIZED);
            }
            if (order != null) {
                List orderItems = order.getItems();
                Customer customer = this.customerStore.getCustomer(order.getCustomer());
                Integer productId = null;
                order.setStatus("trialing");
                order.getStatusTransitions().put("trialing", secondsSinceEpoch);
                order.setStartDate(secondsSinceEpoch);
                order.setEndDate(trialEndSecondsSinceEpoch);
                for (OrderItem item : orderItems) {
                    if (!item.getType().equals("sku")) continue;
                    productId = item.getParent();
                    Product product = this.productStore.getProduct(productId);
                    LinkedHashMap<String, Quota> quotas = new LinkedHashMap<String, Quota>();
                    Iterator featuresIterator = product.getMetadata().get("features").elements();
                    Quota quota = null;
                    Quota existingQuota = null;
                    while (featuresIterator.hasNext()) {
                        ObjectNode featureNode = (ObjectNode)featuresIterator.next();
                        Feature feature = (Feature)this.mapper.readValue(featureNode.toString(), Feature.class);
                        quota = feature.getQuota();
                        if (quota == null) continue;
                        quota.setTotalUsage(Double.valueOf(0.0));
                        quota.setSubject(order.getSubject());
                        quota.setOrderId(order.getId());
                        quota.setName(order.getName());
                        if (!quotas.containsKey(quota.getQuotaType())) {
                            quotas.put(quota.getQuotaType(), quota);
                            continue;
                        }
                        existingQuota = (Quota)quotas.get(quota.getQuotaType());
                        Double newSoftLimit = existingQuota.getSoftLimit() + quota.getSoftLimit();
                        quota.setSoftLimit(newSoftLimit);
                        Double newHardLimit = existingQuota.getHardLimit() + quota.getHardLimit();
                        quota.setHardLimit(newHardLimit);
                        quotas.put(quota.getQuotaType(), quota);
                    }
                    for (Quota newQuota : quotas.values()) {
                        this.quotaStore.insert(newQuota);
                    }
                }
            } else {
                String message = "Couldn't find the order for order id " + orderId;
                throw new WebApplicationException(message, Response.Status.NOT_FOUND);
            }
            order.setUpdated(new Integer((int)Instant.now().getEpochSecond()));
            this.orderStore.update(order);
        }
        catch (Exception e) {
            String message = "Couldn't pay the order: " + e.getMessage();
            throw new WebApplicationException(message, Response.Status.INTERNAL_SERVER_ERROR);
        }
        return order;
    }

    @Timed
    @DELETE
    @PermitAll
    @Path(value="{orderId}")
    public Response delete(@Context SecurityContext context, @PathParam(value="orderId") @Valid Integer orderId) throws WebApplicationException {
        Customer caller = (Customer)context.getUserPrincipal();
        if (!this.dataoneAuthHelper.isBookkeeperAdmin(caller.getSubject())) {
            throw new WebApplicationException("Bookkeeper admin privilege is required to delete an order, " + caller.getSubject() + " is not authorized.", Response.Status.FORBIDDEN);
        }
        String message = "The orderId cannot be null.";
        if (orderId == null) {
            throw new WebApplicationException(message, Response.Status.BAD_REQUEST);
        }
        try {
            this.orderStore.delete(orderId);
        }
        catch (Exception e) {
            message = "Deleting the order with id " + orderId + " failed: " + e.getMessage();
            this.log.error((Object)message);
            e.printStackTrace();
            throw e;
        }
        return Response.ok().build();
    }
}

